/*
 * Function.java
 *
 * Created on October 26, 2004, 10:19 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr;

/** A functionoid designed to implement a scalar function of a scalar
 *
 * @author  Daniel Wachsstock
 */
abstract public class Function {

    /** a scalar function of a scalar.
     *  @param x the independent variable
     *  @return the value
     */
    abstract public double eval (double x);
    
    
    /** The computational cost of evaluating the derivative.
     *  Some algorithms make decisions on how hard a calculation is,
     *  so this function should return the ratio of computer time
     *  in calling {@link #derivative} to {@link #eval}. For the most
     *  part this can be ignored.
     *  <p>
     *  For the default derivative, derivativeCost = 2 (derivative calls
     *  eval twice).
     *  @return the relative cost of evaluating the derivative.
     */
    public double derivativeCost() { return 2d; }
    
    /** Calculates the derivative for this function.
     *  In the default implementation, it is calculated by 
     *  symmetric finite differencing, as in Numerical Recipes
     *  eq. 5.7.7.
     *  This involves 2 calls to eval.
     *  @param x the independent variable
     *  @return df/dx evaluated at x
     */
    public double derivative (double x){
        double oldX = x;
        double h = oldX * EPSILON;
        if (h == 0) h = EPSILON;
        x = oldX - h;
        double h1 = oldX - x; // trick to avoid roundoff errors
          // see Numerical Recipes eq. 5.7.4
        double f1 = eval (x); // find the x-h function values
        x = oldX + h;
        double h2 = x - oldX; // trick to avoid roundoff errors
          // see Numerical Recipes eq. 5.7.4
        double f2 = eval (x); // find the x+h function values
        // now calculate the derivatives
        return  (f2-f1)/(h1+h2);
   } // derivative
    
    static final double EPSILON = 1e-5; // double-precision accuracy ^ 1/3
      // see Numerical Recipes eq. 5.7.8
} // Function
