package nr;

   /** LU Decomposition.
    *
    *  Copied from the <a href="http://math.nist.gov/javanumerics/jama">jama</a>
    *  package from the NIST, but using nr.Mat and limited to
    *  square matrices
    *  <P>
    *  For an n-by-n matrix A, the LU decomposition is an n-by-n
    *  unit lower triangular matrix L, an n-by-n upper triangular matrix U,
    *  and a permutation vector piv of length n so that A(piv,:) = L*U.
    *  <P>
    *  The LU decompostion with pivoting always exists, even if the matrix is
    *  singular, so the constructor will never fail.  The primary use of the
    *  LU decomposition is in the solution of square systems of simultaneous
    *  linear equations.  This will fail if isNonsingular() returns false.
    *
    *  Since this was developed by the US government, it is in the public domain.
    */

public class LUDecomposition implements MatDecomposition {
    private Mat _A;  // the original matrix, decomposed into lower and
      // upper matrices
    private int _n; // _A.size(), for convenience
    private int[] _pivot; // array of row permutations
    private boolean _even; // true if an even number of interchanges
      // was done,; false if odd

   /**  LU Decomposition
    *   @param  A A square matrix
    */
   public LUDecomposition (Mat A) {

   // Use a "left-looking", dot-product, Crout/Doolittle algorithm.

        _A = A;
        _n = A.size();
        _pivot = new int[_n];
        // we do implicit scaling to find the pivot, so we store the
        // largest value in each row (this is added from Numerical Recipes, section 2.3)
        double[] scale = new double[_n];
        for (int r = 0; r < _n; r++) {
            _pivot [r] = r;
            scale [r] = 0.0;
            for (int c = 0; c < _n; c++)
                scale[r] = Math.max(scale[r], Math.abs(_A.get(r,c)));
        } // for r
        _even = true ;

        // Outer loop.

        for (int j = 0; j < _n; j++) {

            // Apply previous transformations.
            for (int i = 0; i < _n; i++) {
                // Most of the time is spent in the following dot product.
                int kmax = Math.min(i,j);
                double s = _A.get(i,j);
                for (int k = 0; k < kmax; k++) {
                    s -= _A.get(i,k)*_A.get(k,j);
                }
                _A.set(i, j, s);
            }
   
            // Find pivot and exchange if necessary.
            // find the pivot--the largest scaled element
            // and exchange if necessary
            int maxIndex = j; // the row with the pivot
            double max = Math.abs (_A.get(j, j) / scale[j]);
            for (int r = j+1; r < _n; r++){
                double scaled = Math.abs(_A.get (r, j) / scale[r]);
                if (scaled >= max){
                    max = scaled;
                    maxIndex = r;
                } // if
            } // for
            if (maxIndex != j) {
                for (int k = 0; k < _n; k++) {
                    // swap
                    double t = _A.get(maxIndex,k);
                    double u = _A.get(j,k);
                    _A.set (maxIndex, k, u);
                    _A.set (j, k, t);
                } // for k
                int k = _pivot[maxIndex];
                _pivot[maxIndex] = _pivot[j];
                _pivot[j] = k;
                _even = ! _even;
            } // if p != j

            // Compute multipliers.
         
            if (j < _n & _A.get(j, j) != 0.0) {
                for (int i = j+1; i < _n; i++) {
                    _A.set (i, j, _A.get(i,j)/_A.get(j,j));
                } // for i
            } // if
        } // for j 
    } // constructor

   /**  Is the matrix nonsingular?
    *   @return true if U, and hence A, is nonsingular.
    */
   public boolean isNonsingular () {
      for (int j = 0; j < _n; j++) {
         if (_A.get(j,j) == 0)
            return false;
      }
      return true;
   } // isNonsingular

   /**  Returns the lower triangular factor
    *   @return     L
    */
   public Mat getL() {
      Mat X = new Mat_array(_n);
      for (int i = 0; i < _n; i++) {
         for (int j = 0; j < _n; j++) {
            if (i > j) {
                X.set(i,j,_A.get(i,j));
            } else if (i == j) {
                X.set (i,j,1.0);
            } else {
                X.set (i,j,0.0);
            }
         }
      }
      return X;
   } // getL

   /**  Returns the upper triangular factor
    *   @return     U
    */
   public Mat getU (){
      Mat X = new Mat_array(_n);
      for (int i = 0; i < _n; i++) {
         for (int j = 0; j < _n; j++) {
            if (i <= j) {
                X.set (i,j,_A.get(i,j));
            } else {
                X.set (i,j,0.0);
            }
         }
      }
      return X;
   } // getU

   /**  Returns the pivot permutation vector
    *   @return  pivot array
    */
   public int[] getPivot () {
      int[] p = new int[_n];
      for (int i = 0; i < _n; i++) {
         p[i] = _pivot[i];
      }
      return p;
   } // getPivot

   /**  Returns the pivot permutation vector as a one-dimensional double array
    *   @return     (double) pivot array
    */
   public double[] getDoublePivot () {
      double[] vals = new double[_n];
      for (int i = 0; i < _n; i++) {
         vals[i] = (double) _pivot[i];
      }
      return vals;
   } // getDoublePivot

   /** Returns the determinant
    *  @return     det(A)
    */
    public double determinant () {
       double d = _even ? 1.0 : -1.0 ;
          for (int j = 0; j < _n; j++) {
             d *= _A.get(j,j);
          }
          return d;
    } // determinant

   /** Solve A*X = B
       @param  B   a Vec
       @return X so that L*U*X = B(with pivoting)
       @throws DidNotConvergeException  if B is singular.
   */

    public Vec solve (Vec B) throws DidNotConvergeException{
        if (!isNonsingular()){
            throw new DidNotConvergeException();
        }

        // unpivot B
        double[] b = new double[_n];
        for (int i = 0; i < _n; i++) b[i] = B.get(_pivot[i]);
        
        // Solve L*Y = b
        for (int c = 0; c < _n; c++){
            for (int r = c + 1; r < _n; r++)
                b[r] -= _A.get (r, c) * b[c];
        } // for
        
        // Solve U*X = Y;
        for (int c = _n - 1; c >= 0; c--){
            b[c] /= _A.get(c,c);
            for (int r = 0; r < c; r++)
                b[r] -= _A.get (r, c) * b[c];
        } // for

        B.set(b);
        return B;
   }
   
    /** find the inverse by solving for each column of the identity matrix.
     *  @returns the inverse of the original matrix
     */
    public Mat inverse(){
        Mat result = Mat_array.identity (_n);
        for (int c = 0; c < _n; c++){
            solve (result.getColumn (c));
        } // for
        return result;
    } // inverse
  
} // LUDecomposition
