/*
 * Mat_array.java
 *
 * Created on July 18, 2004, 9:48 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr;

/** Implements a simple Mat based on a fixed array
 *
 *  @author  Daniel Wachsstock
 */
public class Mat_array implements Mat {
    final double[][] _data;
    
    /** Creates a new instance of Mat_array
     *  @param n the size of the matrix
     */
    public Mat_array(int n) {
        _data = new double[n][];
        for (int i = 0; i < _data.length; i++)
            _data[i] = new double[n];
    } // constructor
    
    /** Creates a new instance of Mat_array from a double[][].
     *  @param d the source array
     *  @throws IndexOutOfBoundsException if d is not a square array
     */
    public Mat_array (double[][] d){
        _data = (double[][]) d.clone();
        for (int i = 0; i < _data.length; i++)
            _data[i] = (double[]) d[i].clone();
    } // constructor
    
    /** Creates a new instance of Mat_array.
     *  Copy constructor.
     *  @param m the Mat to copy
     */
    public Mat_array(Mat m){
        _data = m.asArray();
    } // constructor
    
    public double[][] asArray() {
        double[][] result = (double[][]) _data.clone();
        for (int i = 0; i < _data.length; i++)
            result[i] = (double[]) _data[i].clone();
        return result;
    }
    
    public Mat copy() {
        return new Mat_array(this);
    }
    
    public double get(int i, int j) {
        return _data[i][j];
    }
    
    public Vec getColumn(int j) {
        return new ColumnSlice(j);
    }
    
    public Vec getRow(int i) {
        return new RowSlice(i);
    }
    
    public void set(int i, int j, double d) {
        _data[i][j] = d;
    }
    
    public int size() {
        return _data.length;
    }
    
    /** returns an identity matrix
     *  @param n the size
     *  @return an identity matrix (A[i][j] = i==j ? 1 : 0)
     */
    static public Mat identity (int n){
        Mat_array result = new Mat_array(n);
        for (int i = 0; i < n; i++) result._data[i][i] = 1d;
        return result;
    } // identity
    
    public String toString(){
        String result = "{";
        for (int i = 0; i < _data.length; i++){
            if (i > 0) result += ", ";
            result += getRow(i).toString();
        } // for
        return result+"}";
    }// toString

    
    class RowSlice implements Vec {
        private int _i;
        
        public RowSlice(int i) {_i = i;}      
        
        public double[] asArray() {
            return (double[])_data[_i].clone();
        }
        
        public nr.Vec copy() {
            return new Vec_array(_data[_i]);
        }
        
        public double get(int i) {
            return _data[_i][i];
        }
        
        public void set(int i, double d) {
            _data[_i][i] = d;
        }
        
        public void set (double[] d){
            for (int i = 0; i < d.length; i++) set(i, d[i]);
        }
        
        public void set(Vec v) {
            for (int i = 0; i < v.size(); i++) set(i,v.get(i));
        }

        public int size() {
            return _data.length;
        }
 
    public String toString(){
        String result = "{";
        for (int i = 0; i < _data.length; i++){
            if (i > 0) result += ", ";
            result += _data[_i][i];
        } // for
        return result+"}";
    }// toString
       
    } // RowSlice
    
    class ColumnSlice implements Vec {
        private int _j;
        
        public ColumnSlice (int j) {_j =j ;}
        
        public double[] asArray() {
            double[] result = new double[_data.length];
            for (int i = 0; i < result.length; i++)
                result[i] = _data[i][_j];
            return result;
        }
        
        public nr.Vec copy() {
            return new Vec_array (asArray());
        }
        
        public double get(int i) {
            return _data[i][_j];
        }
        
        public void set(int i, double d) {
            _data[i][_j] = d;
        }
 
        public void set(double[] d) {
            for (int i = 0; i < d.length; i++) set(i, d[i]);
        }

        public void set(Vec v) {
            for (int i = 0; i < v.size(); i++) set(i,v.get(i));
        }
        
        public int size() {
            return _data.length;
        }

    public String toString(){
        String result = "{";
        for (int i = 0; i < _data.length; i++){
            if (i > 0) result += ", ";
            result += _data[i][_j];
        } // for
        return result+"}";
    }// toString
                
    }
}
