/*
Copyright  1999 CERN - European Organization for Nuclear Research.
Permission to use, copy, modify, distribute and sell this software and its documentation for any purpose 
is hereby granted without fee, provided that the above copyright notice appear in all copies and 
that both that copyright notice and this permission notice appear in supporting documentation. 
CERN makes no representations about the suitability of this software for any purpose. 
It is provided "as is" without expressed or implied warranty.
*/
package nr; // copied to package nr by Daniel Wachsstock, 8/23/2005

/**
 *  Polynomial evaluations, from CERN
 */
public class Polynomial{
    /**
     * Makes this class non instantiable, but still lets others inherit from it.
     */
    protected Polynomial() {}

    /**
    *  Evaluates the given polynomial of degree <tt>N</tt> at <tt>x</tt>, assuming coefficient of N is 1.0.
    *  Otherwise same as <tt>polevl()</tt>.
    *  <br>
    *  <code>y = C<sub>0</sub> + C<sub>1</sub>x + C<sub>2</sub>x<sup>2</sup>
    *  + &hellip; + C<sub>N</sub>x<sup>N</sup></code>
    *  <br>
    *  where <code>C<sub>N</sub> = 1</code> and hence is omitted from the array.
    *  <br>
    *  Coefficients are stored in reverse order:
    *  <br>
    *  coef[0] = C<sub>N-1</sub> , hellip, coef[N-1] = C<sub>0</sub>
    *  <br>
    *  Calling arguments are otherwise the same as polevl().
    *  In the interest of speed, there are no checks for out of bounds arithmetic.
    *
    *  @param x argument to the polynomial.
    *  @param coef the coefficients of the polynomial.
    *  @returns the value of the polynomial at x
    */
   public static double eval1( double x, double coef[] ){
       int n = coef.length;
       if (n == 0) return 0;
       double ans = x + coef[0];
       for(int i=1; i<n; i++) ans = ans*x+coef[i];
	 return ans;
   } // eval1

    /**
     *  Evaluates the given polynomial of degree <tt>N</tt> at <tt>x</tt>.
     *  <br>
     *  <code>y = C<sub>0</sub> + C<sub>1</sub>x + C<sub>2</sub>x<sup>2</sup>
     *  + &hellip; + C<sub>N</sub>x<sup>N</sup></code>
     *  <br>
     *  Coefficients are stored in reverse order:
     *  <br>
     *  coef[0] = C<sub>N</sub>, &hellip;, coef[N] = C<sub>0</sub>
     *  <br>
     *
     *  @param x argument to the polynomial.
     *  @param coef the coefficients of the polynomial.
     *  @param N the degree of the polynomial.
     */
    public static double eval( double x, double coef[] ) throws ArithmeticException {
        int n = coef.length;
        if (n == 0) return 0;
	  double ans = coef[0];
	  for(int i=1; i<n; i++) ans = ans*x+coef[i];
        return ans;
    }  // eval

} // Polynomial
