/*
 * PolynomialExtrapolator.java
 *
 * Created on August 3, 2004, 11:45 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr;

/** Extrapolator based on pzextr in Numberical Recipes section 16.4
 *  @author Daniel Wachsstock
 */
public class PolynomialExtrapolator implements Extrapolator{
    private double[] _xs; // the stored x values
    private Vec[] _d; // the D parameter of the algorithm, generated from the y's
    private int _i; // index of the highest column of the extrapolation tableau so far
    private double _target;
    
    /** Create a new instance of PolynomialExtrapolator with initial
     *  tableau size n
     */
    public PolynomialExtrapolator (int n){
        _xs = new double[n];
        _d = new Vec[n];
        _i = -1;
        _target = 0;
    } // constructor
    
    /** Create a new instance of PolynomialExtrapolator with a default initial
     *  size.
     */
    public PolynomialExtrapolator (){ this(9); }
    
    public void add (double x, Vec y, Vec dy){
        x -= _target; // change variables so we can just extrapolate to zero
        _i++; // one more data point
        if (_i >= _xs.length) growArrays();
        _xs[_i] = x;
        // we need dy internally, even if the caller does not
        if (dy == null){
            dy = new Vec_array(y);
        }else{
            dy.set(y);
        }
        if (_i == 0){
            // just one column
            _d[0] = new Vec_array(y);
        }else{
            Vec c = new Vec_array(y);
            for (int col = 0; col < _i; col++){
                // Numerical Recipes equations 3.1.5
                double denom = _xs[_i-col-1] - x;
                double dFactor = x/denom;
                double cFactor = _xs[_i-col-1]/denom;
                for (int i = 0; i < y.size(); i++){
                    double delta = c.get(i) - _d[col].get(i);
                    _d[col].set (i, dy.get (i));
                    dy.set (i, dFactor*delta);
                    c.set (i, cFactor*delta);
                    y.set (i, y.get (i) + dy.get (i));
                } // for i
            }  // for col
            _d[_i] = new Vec_array(dy);
        } // if
    } // add
    
    public void reset () { _i = -1; }
    public void setTarget (double x) { _target = x; reset(); }
    public double getTarget() { return _target; }
    
    // double the size of the arrays
    private void growArrays(){
        double[] newXs = new double[_xs.length*2+1];
        Vec[] newD = new Vec[_xs.length*2+1];
        for (int i = 0; i < _xs.length; i++){
            newXs[i] = _xs[i];
            newD[i] = _d[i];
        } // for
        _xs = newXs;
        _d = newD;
    }
    
} // PolynomialExtrapolator
