/*
 * ScalarFunction.java
 *
 * Created on October 26, 2004, 10:19 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr;

/** A functionoid designed to implement a scalar function of a vector
 *
 * @author  Daniel Wachsstock
 */
abstract public class ScalarFunction {

    /** a scalar function of a vector.
     *  @param x the independent variable
     *  @return y
     */
    abstract public double eval (Vec x);
    
    
    /** The computational cost of evaluating the gradient.
     *  Some algorithms make decisions on how hard a calculation is,
     *  so this function should return the ratio of computer time
     *  in calling {@link #grad} to {@link #eval}. For the most
     *  part this can be ignored.
     *  This will usually depend on the size of the input vector, so is
     *  unknowable to a generic implementation, so the default implementation
     *  throws an Error. If it is needed, it must be implemented (how to get the
     *  necessary information is up to you).
     *  <p>
     *  For the default gradient, gradCost = 2*x.size() (grad calls
     *  eval twice for every element of the input vector)
     *  @return the relative cost of evaluating the Jacobian
     */
    public double gradCost() { throw new Error("gradCost is not implemented"); }
    
    /** Calculates the gradient for this function.
     *  if f = eval(x) and j = grad(x), then
     *  j[i] = df/dx[j].
     *  In the default implementation, it is calculated by 
     *  symmetric finite differencing, as in Numerical Recipes
     *  eq. 5.7.7.
     *  This involves 2*x.size() calls to eval.
     *  @param x the independent variable
     *  @param g the the gradient of eval. If grad == null, a new Vec
     *  will be created
     *  @return g
     */
    public Vec grad (Vec x, Vec g){
        int n = x.size();
        if (g== null) g = new Vec_array (n);
        for (int j = 0; j < n; j++){
            double oldX_J = x.get(j);
            double h = oldX_J * EPSILON;
            if (h == 0) h = EPSILON;
            x.set (j, oldX_J - h);
            double h1 = oldX_J - x.get(j); // trick to avoid roundoff errors
              // see Numerical Recipes eq. 5.7.4
            double f1 = eval (x); // find the x-h function values
            x.set (j, oldX_J + h);
            double h2 = x.get(j) - oldX_J; // trick to avoid roundoff errors
              // see Numerical Recipes eq. 5.7.4
            double f2 = eval (x); // find the x+h function values
            // now calculate the derivatives
            g.set(j, (f2-f1)/(h1+h2));
            x.set(j, oldX_J);
        } // for j
        return g;
    } // grad
    
    /** Calculates the gradient for this function.
     *  Should have the same result as {@link #grad (Vec, Vec)} with
     *  g == null. The default implementation does just that.
     *  @param x the independent variable
     *  @return the jacobian
     */
    public Vec grad (Vec x){ return grad (x, null); }

    static final double EPSILON = 1e-5; // double-precision accuracy ^ 1/3
      // see Numerical Recipes eq. 5.7.8
} // ScalarFunction
