/*
 * SpecialFunctions.java
 *
 * Created on August 18, 2005
 *
 *  Copyright 2005 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr;

/** Some useful functions
 *  @author Daniel Wachsstock
 */

public class SpecialFunctions {
    // can't be created; static functions only
    private SpecialFunctions(){}
    private static final java.util.Random _rand = new java.util.Random();

    /** random gaussian with a given standard deviation */
    public static double randomGaussian (double dev){
       return _rand.nextGaussian()*dev;
    } // randomGaussian

    /** sum of n random exponentials. The gamma deviate from
     *  Numerical Recipes section 7.3. Adds n random deviates with probability
     *  distribution <code>p(x) = exp(-x)</code>
     *  @param n the number of exponentials to add. A double, for
     *  flexibility with the rest of the program. If n does not represent an integer,
     *  for small n uses floor(n); for large n uses a function approximation that
     *  effectively interpolates between floor(n) and floor(n)+1.
     *  @throws IllegalArgumentException if n < 1
     */
    public static double randomExponential (double n){
        double n1, x, y, result, tan, s, e;
        if (n < 1) throw new IllegalArgumentException();
        if (n < 6) {
            // for small n, just add n log(random())'s.
            // or log (product of n random()'s)
		result = 1.0;
		while ( (n--) > 0 ) result *= _rand.nextDouble();
		result = -Math.log(result);
        }else{
            do{
                do{
                    // tangent of a random angle
                    do{  // random point in a half-unit-circle
                        x = _rand.nextDouble();
                        y = 2d*_rand.nextDouble()-1d;
                    }while (x*x+y*y > 1d);
                    tan = y/x;
                    --n;
                    s = Math.sqrt(2d*n+1d);
                    result = s*tan+n; // Lorentzian distribution
                }while (result <= 0d); // don't accept negative probabilities
                e = (1d+tan*tan)*Math.exp(n*Math.log(result/n)-s*tan);
            }while (_rand.nextDouble() > e); // rejection criterion
        } // if
        return result;
    } // randomExponential

    /** chi square distribution. Probability of a chi-square-distributed
     *  statistic (probability that the sum of the squares of n 
     *  normally distributed random deviates with mean 0 and variance 1
     *  exceeds a value x. The higher x is, the lower chisq is.
     *  @param x the sum-of-squares
     *  @param n the "degrees of freedom": the number of sums-of-squares minus
     *  the number of fitted parameters.
     *  @throws IllegalArgumentException if x < 0 or n <= 0
     *  @throws ArithmeticException if the function cannot be estimated
     */
    public static double chisq (double x, double n){
        return incompleteGammaComplement (n/2d, x/2d);
    }  // chisq


    /* The remainder of the file is subject to the following copyleft:
    Copyright  1999 CERN - European Organization for Nuclear Research.
    Permission to use, copy, modify, distribute and sell this software and its documentation for any purpose 
    is hereby granted without fee, provided that the above copyright notice appear in all copies and 
    that both that copyright notice and this permission notice appear in supporting documentation. 
    CERN makes no representations about the suitability of this software for any purpose. 
    It is provided "as is" without expressed or implied warranty.
    */

    /*
    * machine constants
    */
    protected static final double MACHEP =  1.11022302462515654042E-16;
    protected static final double MAXLOG =  7.09782712893383996732E2;
    protected static final double MINLOG = -7.451332191019412076235E2;
    protected static final double MAXGAM = 171.624376956302725;
    protected static final double SQTPI  =  2.50662827463100050242E0;
    protected static final double SQRTH  =  7.07106781186547524401E-1;
    protected static final double LOGPI  =  1.14472988584940017414;
    protected static final double BIG = 4.503599627370496e15;
    protected static final double BIGINV =  2.22044604925031308085e-16;

    /**
     *  Returns the natural logarithm of the gamma function.
     *  Copied from the CERN COLT package.
     *  n! is exp (logGamma (n+1))
     *  <p>
     *  <b>Implementation:</b>
     *  <dt>
     *  Some code taken and adapted from the
     *  <A HREF="http://www.sci.usq.edu.au/staff/leighb/graph/Top.html">
     *  Java 2D Graph Package 2.4</A>,
     *  which in turn is a port from the
     *  <A HREF="http://people.ne.mediaone.net/moshier/index.html#Cephes">
     *  Cephes 2.2</A> Math Library (C).
     *  Most Cephes code (missing from the 2D Graph Package) directly ported.
     *  @param x the argument to the function
     *  @returns the natural logarithm of the gamma function
     *  @throws ArithmeticException if the function overflows
     *  @author wolfgang.hoschek@cern.ch
     *  @version 0.9, 22-Jun-99
     */
    public static double logGamma(double x) throws ArithmeticException {
        double p, q, w, z;
        double A[] = {
           8.11614167470508450300E-4,
          -5.95061904284301438324E-4,
           7.93650340457716943945E-4,
          -2.77777777730099687205E-3,
           8.33333333333331927722E-2
        };
        double B[] = {
          -1.37825152569120859100E3,
          -3.88016315134637840924E4,
          -3.31612992738871184744E5,
          -1.16237097492762307383E6,
          -1.72173700820839662146E6,
          -8.53555664245765465627E5
        };
        double C[] = {
        /* 1.00000000000000000000E0, */
          -3.51815701436523470549E2,
          -1.70642106651881159223E4,
          -2.20528590553854454839E5,
          -1.13933444367982507207E6,
          -2.53252307177582951285E6,
          -2.01889141433532773231E6
        };
        if( x < -34.0 ) {
            q = -x;
            w = logGamma(q);
            p = Math.floor(q);
            if( p == q ) throw new ArithmeticException("logGamma: Overflow");
            z = q - p;
            if( z > 0.5 ) {
                p += 1.0;
                z = p - q;
            }
            z = q * Math.sin( Math.PI * z );
            if( z == 0.0 ) throw new ArithmeticException("lgamma: Overflow");
            z = LOGPI - Math.log( z ) - w;
            return z;
        }
        if( x < 13.0 ) {
            z = 1.0;
            while ( x >= 3.0 ) {
                x -= 1.0;
                z *= x;
            }
            while ( x < 2.0 ) {
                if ( x == 0.0 ) throw new 
                  ArithmeticException("lgamma: Overflow");
                z /= x;
                x += 1.0;
            }
            if ( z < 0.0 ) z = -z;
            if( x == 2.0 ) return Math.log(z);
            x -= 2.0;
            p = x * Polynomial.eval(x, B) / Polynomial.eval1(x, C);
            return( Math.log(z) + p );
        }
        if ( x > 2.556348e305 ) throw new
          ArithmeticException("lgamma: Overflow");
        q = ( x - 0.5 ) * Math.log(x) - x + 0.91893853320467274178;
        if( x > 1.0e8 ) return( q );
        p = 1.0/(x*x);
        if( x >= 1000.0 ) {
            q += ((   7.9365079365079365079365e-4 * p
                    - 2.7777777777777777777778e-3) * p
                    + 0.0833333333333333333333) / x;
        }else{
            q += Polynomial.eval(p, A) / x;
        }
        return q;
    } // logGamma

    /**
     *  Returns the Incomplete Gamma function; formerly named <tt>igamma</tt>.
     *  @param a the parameter of the gamma distribution.
     *  @param x the integration end point.
     */
    static private double incompleteGamma(double a, double x)
      throws ArithmeticException {
        double ans, ax, c, r;
        if( x <= 0 || a <= 0 ) return 0.0;
        if( x > 1.0 && x > a ) return 1.0 - incompleteGammaComplement(a,x);

        /* Compute  x**a * exp(-x) / gamma(a)  */
        ax = a * Math.log(x) - x - logGamma(a);
        if( ax < -MAXLOG ) return( 0.0 );
        ax = Math.exp(ax);

        /* power series */
        r = a;
        c = 1.0;
        ans = 1.0;
        do {
            r += 1.0;
            c *= x/r;
            ans += c;
        } while( c/ans > MACHEP );
        return( ans * ax/a );
    } // incompleteGamma

    /**
     *  Returns the Complemented Incomplete Gamma function; formerly named <tt>igamc</tt>.
     *  @param a the parameter of the gamma distribution.
     *  @param x the integration start point.
     */
    static private double incompleteGammaComplement( double a, double x )
      throws ArithmeticException {
        double ans, ax, c, yc, r, t, y, z;
        double pk, pkm1, pkm2, qk, qkm1, qkm2;
        if( x <= 0 || a <= 0 ) return 1.0;
        if( x < 1.0 || x < a ) return 1.0 - incompleteGamma(a,x);
        ax = a * Math.log(x) - x - logGamma(a);
        if( ax < -MAXLOG ) return 0.0;
        ax = Math.exp(ax);

        /* continued fraction */
        y = 1.0 - a;
        z = x + y + 1.0;
        c = 0.0;
        pkm2 = 1.0;
        qkm2 = x;
        pkm1 = x + 1.0;
        qkm1 = z * x;
        ans = pkm1/qkm1;
        do {
            c += 1.0;
            y += 1.0;
            z += 2.0;
            yc = y * c;
            pk = pkm1 * z  -  pkm2 * yc;
            qk = qkm1 * z  -  qkm2 * yc;
            if( qk != 0 ) {
                r = pk/qk;
                t = Math.abs( (ans - r)/r );
                ans = r;
            }else{
		    t = 1.0;
            }
            pkm2 = pkm1;
            pkm1 = pk;
            qkm2 = qkm1;
            qkm1 = qk;
            if( Math.abs(pk) > BIG ) {
                pkm2 *= BIGINV;
                pkm1 *= BIGINV;
                qkm2 *= BIGINV;
                qkm1 *= BIGINV;
            }
        }while( t > MACHEP );
        return ans * ax;
    } // incompleteGammaComplement

} // SpecialFunctions