/*
 * VecFunction.java
 *
 * Created on July 18, 2004, 10:19 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr;

/** A functionoid designed to implement a vector function of a vector
 *
 * @author  Daniel Wachsstock
 */
abstract public class VecFunction {

    /** a vector function of a vector.
     *  @param x the independent variable
     *  @param y the resulting dependent vector.
     *  if y == null, a new Vec will be created
     *  @return y
     */
    abstract public Vec eval (Vec x, Vec y);
    
    /** a vector function of a vector.
     *  Should have the same result as {@link #eval(Vec,Vec)} with
     *  y == null. The default implementation just does that.
     *  @param x the independent variable
     *  @return the result
     */
    public Vec eval (Vec x) { return eval(x, null); }
    
    /** The computational cost of evaluating the Jacobian.
     *  Some algorithms make decisions on how hard a calculation is,
     *  so this function should return the ratio of computer time
     *  in calling {@link #jacobian} to {@link #eval}. For the most
     *  part this can be ignored.
     *  This will usually depend on the size of the input vector, so is
     *  unknowable to a generic implementation, so the default implementation
     *  throws an Error. If it is needed, it must be implemented (how to get the
     *  necessary information is up to you).
     *  <p>
     *  For the default jacobian, jacobianCost = 2*x.size() (jacobian calls
     *  eval twice for every element of the input vector).
     *  @return the relative cost of evaluating the Jacobian
     */
    public double jacobianCost() { throw new Error("jacobianCost is not implemented"); }
    
    /** Calculates the Jacobian for this function.
     *  Assumes that eval(x) has the same size as x, though that
     *  is not imposed by the specification of eval.
     *  if f = eval(x) and j = jacobian(x), then
     *  j[i][j] = df[i]/dx[j].
     *  In the default implementation, it is calculated by 
     *  symmetric finite differencing, as in Numerical Recipes
     *  eq. 5.7.7.
     *  This involves 2*x.size() calls to eval.
     *  @param x the independent variable
     *  @param jac the the jacobian of eval. If jac == null, a new Mat
     *  will be created
     *  @return jac
     */
    public Mat jacobian (Vec x, Mat jac){
        int n = x.size();
        if (jac == null) jac = new Mat_array (n);
        Vec f1 = new Vec_array (n);
        Vec f2 = new Vec_array (n);
        for (int j = 0; j < n; j++){
            double oldX_J = x.get(j);
            double h = oldX_J * EPSILON;
            if (h == 0) h = EPSILON;
            x.set (j, oldX_J - h);
            double h1 = oldX_J - x.get(j); // trick to avoid roundoff errors
              // see Numerical Recipes eq. 5.7.4
            eval (x, f1); // find the x-h function values
            x.set (j, oldX_J + h);
            double h2 = x.get(j) - oldX_J; // trick to avoid roundoff errors
              // see Numerical Recipes eq. 5.7.4
            eval (x, f2); // find the x+h function values
            // now calculate the derivatives
            for (int i = 0; i < n; i++){
                jac.set(i, j, (f2.get(i)-f1.get(i))/(h1+h2));
            } // for i
            x.set(j, oldX_J);
        } // for j
        return jac;
    } // jacobian
    
    /** Calculates the Jacobian for this function.
     *  Should have the same result as {@link #jacobian (Vec, Mat)} with
     *  jac == null. The default implementation does just that.
     *  @param x the independent variable
     *  @return the jacobian
     */
    public Mat jacobian (Vec x){ return jacobian (x, null); }

    static final double EPSILON = 1e-5; // double-precision accuracy ^ 1/3
      // see Numerical Recipes eq. 5.7.8
} // VecFunction
