/*
 * VecMinimizerImp.java
 *
 * Created on May 27, 2005, 02:30 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr.minimizer;
import nr.*;

/** Simple implementation of a {@link VecMinimizer}. The constructor takes a
 *  {@link #nr.ScalarFunction} and epsilon is used to create a vector, such that
 *  if the trial minimum is changed less than that vector in each component, the
 *  algorithm has converged.
 *  The concrete implementation needs to implement {@link doMinimize}.
 *  @author  Daniel Wachsstock
 */

abstract public class VecMinimizerImp implements VecMinimizer{
    /** The maximum number of times to evaluate the
     *  function before throwing {@link #nr.DidNotConvergeException}
     */
    public final int MAXFUNCTIONEVAL = 1000;
    
    protected ScalarFunction _f; // the function to minimize
    protected Vec _x; // the current estimate of the minimum
    protected int _n; // the number of dimensions (components of _x);
    protected double _fx; // the value of _f.eval (_x)
    protected int _nf = 0; // the number of calls to _f made so far
    protected double _eps = 1e-3; // the relative accuracy parameter
    protected Vec _minDeltaX; // if each component of _x changes less than the corresponding
      // component of this, then we have converged

    /** create a new instance of VecMinimizerImp.
     *  @param f the {@link nr.ScalarFunction} to minimize
     */
    public VecMinimizerImp (ScalarFunction f){
        _f = f;
    } // constructor

    /** minimize.
     *  @param x the initial guess; epsilon*x is the convergence criterion. 
     *  x will be changed to the minimum of the initial function.
     *  @throws nr.DidNotConvergeException if the algorithm could not find the minimum
     *  @return the value of the function at the minimum.
     */
    public double minimize (Vec x){
        _nf = 0;
        _x = x;
        _n = _x.size();
        _fx = eval (_x);
        // set the convergence criteria
        _minDeltaX = new Vec_array(_n);
        for (int i = 0; i < _n; i++){
            double d = _x.get(i);
            if (d != 0d){
                _minDeltaX.set(i, _eps * Math.abs (d));
            }else{
                _minDeltaX.set(i, _eps);
            } // if
        } // for
        // actually minimize
        doMinimize();
        return _fx;
    } // minimize

    /** the algorithm to actually do the minimization **/
    abstract void doMinimize();

    // return true if the difference between a and b is small enough
    protected boolean converged (Vec a, Vec b) {
        for (int i = 0; i < _n; i++){
            double diff = Math.abs (a.get(i)-b.get(i));
            if (diff > _minDeltaX.get(i)) return false;
        } // for
        return true;
    } // converged

    // evaluate the function, checking the number of times it has been evaluated
    protected double eval (Vec x){
        _nf++;
        if (_nf > MAXFUNCTIONEVAL) throw new DidNotConvergeException();
        return _f.eval(x);
    } // eval

    public double getEpsilon() {return _eps;}
    
    public void setEpsilon (double epsilon) {_eps = epsilon;}
    
    public int numFuncEvals() {return _nf;}

    // prints _x, and _fx
    protected void xDebug(){
        System.out.println("_x = "+_x+"; _fx = "+_fx+" after "+_nf+" evaluations");
    }

} // VecMinimizerImp