/*
 * AbstractBulirschStoer.java
 *
 * Created on August 10, 2004, 12:53 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr.ode;
import nr.*;

/** Implements the Bulirsch-Stoer algorithm for solving ODE's.
 *  Based on Numerical Recipes section 16.4
 *  <p>
 *  Works by calling another ODE solver (the "slave") with smaller and smaller
 *  values of epsilon, then extrapolating to epsilon == 0.
 *  The algorithm assumes that the error is even in epsilon
 *  (only even powers of epsilon are involved in the error estimate
 *  for the slave).
 *  This is true for the modified midpoint and the implicit midpoint methods
 *  given in Numerical Recipes. If the error is not even, the method will
 *  still work but will require much smaller stepsizes to achieve the same
 *  accuracy
 *  @author  Daniel Wachsstock
 */
abstract class AbstractBulirschStoerODE extends ODE{
    
    // the ODE to actually solve the differential equation
    protected ODE _slaveODE;
    // the algorithm to do the extrapolation
    protected Extrapolator _extrapolator;
    // safety factor; reduce stepsize by .25 what we would otherwise predict
    // See comment after Numerical Recipes equation 16.4.17
    static final private double SAFE1 = 0.25;

    // the variables starting with k represent successive attempts to do the
    // step, that are all extrapolated together to get the final answer
    protected int _kMax; // the maximum number of attempts before just shrinking
      // the step size
    protected int _kOpt; // the predicted "optimal" number of attempts--the
      // point where the algorithm thinks making more attempts would be too
      // much work, with "work" defined as the number of calls to _dxdt.eval()
    
    // _a[k] is the work for a step with epsilon = substepEpsilon(k)
    // Numerical Recipes equation 16.4.6
    protected double[] _a = null; 
    // _alpha[k][q] is the Deuflhard convergence factor for attempt k in
    // the extrapolation tableau column q. See Numerical Recipes equation 16.4.10
    // defined only for k < q
    protected double[][] _alpha = null;
    // both _a and _alpha are lazy-instantiated because they depend on numEpsilons()
    // which is abstract--the subclass's version needs to be called, so it cannot be
    // called from this class's constructor.
    
    /** Creates a new instance of BulirschStolerODE */
    public AbstractBulirschStoerODE (VecFunction dxdt) {
        super(dxdt);
    } // constructor
    
    // Epsilon determines the relative convergence factors, so we recalculate them
    // every time epsilon is set
    public void setEpsilon (double epsilon){
        super.setEpsilon(epsilon);
        if (_a == null) _a = new double[numEpsilons()-1];
        if (_alpha == null) _alpha = new double[numEpsilons()-1][numEpsilons()-1];
        _kMax = numEpsilons()-2; // maximum index into _a
        // calculate the work in solving the ODE with a given epsilon
        _a[0] = cost (substepEpsilon(0));
        // each subsequent step involves doing all the previous step, plus the work of this one
        for (int k = 0; k < _kMax; k++) _a[k+1] = _a[k] + cost (substepEpsilon (k));
        // calculate the Deuflhard coefficients (from Numerical Recipes equation 16.4.10
        for (int q = 1; q < _kMax; q++) for (int k = 0; k < q; k++){
            _alpha[k][q] =
              Math.pow (SAFE1*_eps, (_a[k+1]-_a[q+1]) / ((_a[q+1]-_a[0]+1)*(2*k+3)));
        } // for k, q
        // calculate the maximum k to use
        for (int _kOpt = 1; _kOpt < _kMax-1; _kOpt++){
            if (_a[_kOpt+1] > _a[_kOpt]*_alpha[_kOpt-1][_kOpt]){
                _kMax = _kOpt;
                break;
            } // if
        } // for
    } // setEpsilon

    void solveStep(Vec x, Vec dxdt, double[] h, Vec scale)
      throws DidNotConvergeException {
        if (_a == null || _alpha == null) setEpsilon (_eps);
        Vec xNew = new Vec_array (x.size());
        Vec xErr = new Vec_array (x.size());
        double err[] = new double[numEpsilons()-1];
        for (int trial = 0; trial < MAX_ITERATIONS; trial++){
            // allow for multi-threading
            Thread.yield();
            if (Thread.interrupted()){
                Thread.currentThread().interrupt(); // re-raise
                return;
            } // if
            int k; // the current attempt number
            boolean reduced = false; // true if we needed to reduce the step size
            double red = 0; // the smallest factor to reduce by
            _extrapolator.reset();
            for (k = 0; k <= _kMax; k++){
                extrapolate (x, h, xNew, xErr, k);
                if (k == 0) continue; // need at least 2 extrapolations to calculate errors
                double errMax  = maxRatio (xErr, scale, 0) / _eps;
                err [k-1] = Math.pow (errMax/SAFE1, 1/(2*k+1));
                if (errMax < 1) {
                    // converged !
                    h[1] = h[0];
                    h[0] = calculateNewH (h[0], k, err, reduced); // also sets _kOpt for the next call
                    x.set (xNew);
                    return;
                } // if
                red = calculateRed (k, err);
                if (red != 0) break; // we decided to reduce
            } // for k
            if (red == 0) throw new Error ("red==0 in solve; is a bug");
            // reduce the stepsize
            h[0] *= red;
            reduced = true;
        } // for trial
        throw new DidNotConvergeException();
    } // solveStep
    
    /** calculates the relative computer cost of evaluating at a given epsilon
     *  in the slave ODE
     *  @param epsilon the hypothetical epsilon to try
     *  @return the cost (the number of calls to the derivative-calculating
     *  function)
     */
    abstract protected double cost (double epsilon);

    // the epsilon to try in the j-th attempt
    abstract protected double substepEpsilon (int j);
    
    // the maximum number of epsilons to try
    abstract protected int numEpsilons();
    
    // solve the ODE for a given epsilon and add it to the extrapolation tableau
    protected void extrapolate (Vec x, double[] h, Vec xNew, Vec xErr, int k){
        xNew.set (x);
        double epsilon = substepEpsilon (k);
        _slaveODE.setEpsilon (epsilon);
        _slaveODE.solve (xNew, h[0]);
        _extrapolator.add (epsilon*epsilon, xNew, xErr);
    } // extrapolate 

    // calculate the new step size and optimal k
    // from the end of Numerical Recipes bsstep
    double calculateNewH (double h, int k, double[] err, boolean reduced){
        final double SCALE_MAX = 0.1; // 1/the maximum factor to increase by
        double workMin = Double.MAX_VALUE;
        double scale = SCALE_MAX;
        for (int kk = 0; kk < k-1; kk++){
            double tryScale = Math.max (err[kk], SCALE_MAX);
            double work = tryScale * _a[kk+1];
            if (work < workMin) {
                scale = tryScale;
                workMin = work;
                _kOpt = kk+1;
            } // if
        } // for
        double result = h/scale;
        if (_kOpt >= k && _kOpt != _kMax && ! reduced){
            scale = Math.max (scale/_alpha[_kOpt-1][_kOpt], SCALE_MAX);
            if (_a[_kOpt+1] * scale <= workMin){
                result = h/scale;
                ++_kOpt;
            } // if
        } // if
        return result;
    } // calculateNewH

    // calculate the factor to reduce the step size by, if any.
    // based on the middle section of Numerical Recipes bsstep
    // Does not check to see if this is the first with these parameters
    // the way bsstep does, so it may be a bit less efficient
    double calculateRed (int k, double[] err){
        final double SAFETY = 0.7; // factor to reduce step size even more
        double result = 0;
        if (k == _kMax || k == _kOpt+1)
            // this is the test for the final pass; if none of the others
            // pass on previous trials we will eventually have
            // k == _kMax and will have to reduce.
            result = SAFETY;
        else if (k == _kOpt && _alpha[_kOpt-1][_kOpt] < err[k-1])
            result = 1;
        else if (_kOpt == _kMax && _alpha[k-1][_kMax-1] < err[k-1])
            result = SAFETY * _alpha[k-1][_kMax-1];
        else if (_alpha[k-1][_kOpt] < err[k-1])
            result = _alpha[k-1][_kOpt-1];
        else // don't reduce; try again.
            return 0;
        result /= err[k-1];
        result = Math.max (Math.min (result, 0.7), 1e-5); // force to 1e-5 < result < 0.7
        return result;
    } // calculateRed

    public static String name(){
        return "Abstract Bulirsch-Stoer";
    }
    
    public static String description(){
        return "Should never be instantiated";
    }

} // AbstractBulirschStoerODE
