/*
 * BaderDeuflhardODE.java
 *
 * Created on August 10, 2004, 2:57 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr.ode;
import nr.*;

/** the Bader-Deuflhard algorithm, as given in Numerical Recipes, section 16.6.
 *  Used for stiff equations 
 *  Uses a SemiImplicitMidpointODE as the slave and polynomial extrapolation for 
 *  the extrapolator.
 *  Based on stifbs from Numerical Recipes
 */
class BaderDeuflhardODE extends AbstractBulirschStoerODE {
    // the sequence of step-sizes from Numerical Recipes equation 16.6.35
    private final double[] _epsilons = {1d/2, 1d/6, 1d/10, 1d/14, 1d/22, 1d/34,
      1d/50, 1d/70};
    // another pointer to the slave ODE, that knows the details of
    // the algorithm. I do this because I hate to downcast
    private final SemiImplicitMidpointODE _simo;
    
    public BaderDeuflhardODE(VecFunction dxdt){
        super (dxdt);
        _slaveODE = _simo = new SemiImplicitMidpointODE (dxdt);
        _extrapolator = new PolynomialExtrapolator (numEpsilons());
    } // constructor

    void solveStep(Vec x, Vec dxdt, double[] h, Vec scale)
      throws DidNotConvergeException {
          _simo.setJacobian (_dxdt.jacobian(x));
          super.solveStep(x, dxdt, h, scale);
    }

    protected double cost (double epsilon){
        // would take 1/epsilon substeps, each with one call to _dxdt.eval
        // plus the time = 0 call.
        // the first step (with the lowest epsilon) also calculates the Jacobian
        if (epsilon == _epsilons[0]) return 3 + _dxdt.jacobianCost();
        return 1/epsilon + 1;
    }
    
    protected double substepEpsilon (int j){
        return _epsilons[j];
    } // substepEpsilon

    protected int numEpsilons() { return _epsilons.length; }

    public static String name(){
        return "Bader-Deuflhard";
    }
    
    public static String description(){
        return "Tries a modified-midpoint method (a stiff equation solver) "+
        "multiple times with " +
        "increasing accuracy and extrapolates "+
        "to infinite accuracy";
    }
} // BaderDeuflhardODE

