/** Embedded Runge-Kutta; algorithm from Numerical Recipes section 16.2
 *  with adaptive step sizes */
package nr.ode;
import nr.*;

class EmbeddedRungeKutta extends ODE {

    public EmbeddedRungeKutta (VecFunction dxdt){
        super (dxdt);
    }
    
    void solveStep(Vec x, Vec dxdt, double[] h, Vec scale)
      throws DidNotConvergeException {
        Vec xTemp = new Vec_array(x.size());
        Vec xErr = new Vec_array(x.size());
        double errorEstimate;

        for(int i = 0; i < MAX_ITERATIONS; i++){
            rkck (x, dxdt, h, xTemp, xErr); // try the step
            errorEstimate = maxRatio (xErr, scale, 0) / _eps;
            if (errorEstimate <= 1.0d){ // done!
                x.set (xTemp);
                h[1] = h[0]; // step size we used
                // next step size. Grow it, but no more than a factor of 5
                // ERRCON is (5/SAFETY) ^ (1/PGROW), so we don't calculate the power
                // if we don't need to
                if (errorEstimate > ERRCON){
                    h[0] *= SAFETY * Math.pow(errorEstimate,PGROW);
                }else{
                    h[0] *= 5.0;
                }
                return;
            } // if
            // reduce stepsize, but no more than by 0.1
            double htemp = SAFETY * h[0] * Math.pow(errorEstimate, PSHRINK);
            if (h[0] > 0){
                h[0] = Math.max(htemp, 0.1 * h[0]);
            }else{
                h[0]= Math.min (htemp, 0.1 * h[0]);
            }
            Thread.yield();
            if (Thread.interrupted()){
                // stop calculating and pass the interruption up
                Thread.currentThread().interrupt();
                return;
            } // if
        } // for
        throw new DidNotConvergeException();
    } // solveStep

    /** fifth-order Runge-Kutta step, plus an error estimate */    
    private void rkck (Vec x, Vec dxdt, double[] h, Vec xTemp, Vec xErr){
        // will need to evaluate dxdt at six places: start, 4 intermediate and at the end
        // start, we are given in dxdt. The rest we will place in dxdtArray
        Vec[] dxdtArray = new Vec[6];  
        int i;

        // first step
        dxdtArray[0] = dxdt;
        for (i = 0; i < x.size(); i ++){
            xTemp.set (i, x.get(i) + h[0] * (b[0][0] * dxdtArray[0].get(i)));
        }
        // second step
        dxdtArray[1] = _dxdt.eval(xTemp);
        for (i = 0; i < x.size(); i ++){
            xTemp.set (i, x.get(i) +
              h[0]*(b[1][0]*dxdtArray[1].get(i)
                   +b[1][1]*dxdtArray[0].get(i)));
        }
        // third step
        dxdtArray[2] = _dxdt.eval(xTemp);
        for (i = 0; i < x.size(); i ++){
            xTemp.set (i, x.get(i) +  
              h[0]*(b[2][0]*dxdtArray[0].get(i) +
                    b[2][1]*dxdtArray[1].get(i) +
                    b[2][2]*dxdtArray[2].get(i)));
        }
        // fourth step
        dxdtArray[3] = _dxdt.eval(xTemp);
        for (i = 0; i < x.size(); i ++){
            xTemp.set (i, x.get(i) +
              h[0]*(b[3][0]*dxdtArray[0].get(i) +
                    b[3][1]*dxdtArray[1].get(i) +
                    b[3][2]*dxdtArray[2].get(i) +
                    b[3][3]*dxdtArray[3].get(i)));
        }
         // fifth step
        dxdtArray[4] = _dxdt.eval(xTemp);
        for (i = 0; i < x.size(); i ++){
            xTemp.set (i, x.get(i) +
              h[0]*(b[4][0]*dxdtArray[0].get(i) +
                    b[4][1]*dxdtArray[1].get(i) +
                    b[4][2]*dxdtArray[2].get(i) +
                    b[4][3]*dxdtArray[3].get(i) +
                    b[4][4]*dxdtArray[4].get(i)));
        }
        // sixth and final step
        dxdtArray[5] = _dxdt.eval(xTemp);
        for (i = 0; i < x.size(); i ++){
            xTemp.set (i, x.get(i) +
              h[0]*(c[0]*dxdtArray[0].get(i) +
                    c[2]*dxdtArray[2].get(i) +
                    c[3]*dxdtArray[3].get(i) +
                    c[5]*dxdtArray[5].get(i)));
        }
        // evaluate error estimate
        for (i = 0; i < x.size(); i++){
            xErr.set (i,  
              h[0]*(d[0]*dxdtArray[0].get(i) +
                    d[2]*dxdtArray[2].get(i) +
                    d[3]*dxdtArray[3].get(i) +
                    d[4]*dxdtArray[4].get(i) +
                    d[5]*dxdtArray[5].get(i)));
        }
    } // rkck

    // the constants. Note that 
    // we don't use a[], since _dxdt.eval is time-independent.
    // The method still works.
    final static double[][] b =
    {{1d/5},
     {3d/40, 9d/40},
     {3d/10, -9d/10, 6d/5},
     {-11d/54, 5d/2, -70d/27, 35d/27},
     {1631d/55296, 175d/512, 575d/13824, 44275d/110592, 253d/4096}};
    final static double[] c = {37d/378, 0, 250d/621, 125d/594, 0, 512d/1771};
    final static double[] d = {c[0]-2825d/27648, 0, c[2]-18575d/48384, 
      c[3]-13525d/55296, -277d/14336, c[5]-0.25d};

    // more constants
    final static double SAFETY = 0.9;
    final static double PGROW = -0.2;
    final static double PSHRINK = -0.25;
    final static double ERRCON = 1.89e-4;

    public static String name(){ return "Embedded Runge Kutta";}
    public static String description(){
        return "A variation on Runge-Kutta that automatically changes " +
          "the size of the sub-step " +
          "to keep the total estimated fractional error less than"  +
          "epsilon, which should be " +
          "something like 1e-4.";
    } // description
    
} // EmbeddedRungeKutta
