/** Implements the modified midpoint algorithm for solving ODE's.
 *  From Numerical Recipes section 16.3.  Takes a fixed number of substeps
 *  determined as 1/epsilon.
 */
package nr.ode;
import nr.*;

class MidpointODE extends ODE{
    
    public MidpointODE(VecFunction dxdt){
        super(dxdt);
    } // constructor
        
    void solveStep(Vec x, Vec dxdt, double[] h, Vec scale)
      throws DidNotConvergeException {
	int n=x.size();
	Vec x1 = new Vec_array(n);
        Vec x2 = new Vec_array(n);
        int numSteps = (int) (1d/_eps);
        if (numSteps < 2) numSteps = 2;
	double stepSize = h[1] / numSteps;
	for (int i=0; i < n; i++) {
            x1.set (i, x.get(i));
            x2.set (i, x.get(i) + stepSize * dxdt.get(i));
	} // for
	_dxdt.eval(x2, dxdt);
	for (int step = 1; step < numSteps; step++) {
            for (int i = 0; i < n; i++){
                double swap = x1.get(i) + 2 * stepSize * dxdt.get(i);
                x1.set (i, x2.get(i));
                x2.set (i, swap);
            } // for i
            _dxdt.eval (x2, dxdt);
	} // for step
	for (int i = 0; i < n; i++){
            x.set (i, 0.5 * (x1.get(i) + x2.get(i) + stepSize * dxdt.get(i)));
        } // for i
    } // solveStep
        
    public static String name(){
        return "Modified Midpoint";
    }
    
    public static String description(){
        return "Uses the trapezoidal rule to integrate an ODE. Uses a sub-stepsize of "+
          "timeStep/epsilon and evaluates the derivative at each point. A very fast but "+
          "inaccurate method. Use epsilon 0.1-.001 for best results.";
    }
} // MidpointODE