/*
 * ODE.java
 *
 * Created on July 20, 2004, 2:46 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package nr.ode;
import nr.*;

/** solves one step of an Ordinal Differential Equation.
 *  The constructor takes a vector function that returns dx/dt for
 *  a given x (note the notation here is different from Numerical Recipes
 *  chapter 16, which uses dy/dx as a function of x.
 *  The solver also takes
 *  a desired accuracy, epsilon. The interpretation of epsilon is
 *  entirely dependent on the implementation of this abstract class
 *  <p>
 *  Each call to the method solve(x,h) updates the vector x with the new values
 *  time h later.
 *  <p>
 *  Assumes dx/dt is independent of t (is a function only of x).
 *  If dxd/t is time-dependent, add a new variable to
 *  x with x[n] = start time on input and dxdt[n] = 1 always.
 * *
 * @author  Daniel Wachsstock
 */
public abstract class ODE {
    protected final VecFunction _dxdt;
    protected double _eps = 1e-3; // default value
    
    // the maximum number of substeps to try
    // anything more, and solve throws DidNotConvergeException
    final public static int MAX_ITERATIONS = 10000;
    
    /** Creates a new instance of ODE
     *  @param dxdt the VecFunction that describes this ode
     */
    public ODE (VecFunction dxdt) {
        _dxdt = dxdt;
    }
    
    /** set the desired relative accuracy parameter. Each
     *  implementation decides how to interpret this, if at all
     */
    public void setEpsilon (double epsilon) {_eps = epsilon;}
    
    /** return the relative accuracy parameter
     */
    public double getEpsilon () {return _eps;}
    
    /** solve the ODE starting from x at time 0 to time endT, updating x
     *  @param x the starting vector. On return, the ending vector
     *  @param endT the ending time. endT is allowed to be negative.
     *  @throws DidNotConvergeException if the solver could not
     *  converge on a solution
     */
    public void solve (Vec x, final double endT) throws DidNotConvergeException{

        Vec dxdt = new Vec_array(x.size());
        Vec scale = new Vec_array(x.size());
        // h is the substep time (or deltaT); h[0] is the time step to
        // try and h[1] is the maximum to allow.
        // On return from {@link #solveStep} h[0] is the step to try next time
        // and h[1] is the step actually used.
        // We start by trying to go the whole distance
        double[] h = {endT, endT};
        // the current time
        double t = 0;
  
        for (int steps = 0; steps < MAX_ITERATIONS; steps++){
            _dxdt.eval(x, dxdt);
            // scale is from Numerical Recipes, p.718
            // they add an amount TINY to this to prevent division by zero,
            // but the routine maxRatio already watches for this
            for (int i = 0; i < x.size(); i++)
                scale.set(i, Math.abs(x.get(i))+Math.abs(dxdt.get(i)*h[0]));
            h[1] = endT - t; // maximum step size
            if (Math.abs(h[0]) > Math.abs(h[1])) h[0] = h[1]; // don't overshoot
            solveStep (x, dxdt, h, scale);
            double oldT = t;
            t += h[1];
            if (t == oldT) throw new DidNotConvergeException();
            if (t >= endT) return;
            // now allow for multithreading
            Thread.yield();
            // check for interruption, and reraise the
            // interrupt so the calling methods can get
            // interrupted too.
            if (Thread.interrupted()) {
                Thread.currentThread().interrupt();
                return;
            }
        } // for

        // if we get here, we exceeded MAXITERATIONS steps
        throw new DidNotConvergeException();
    } // solve
    
    /** solve the ODE starting from x at time 0 to time less than
     *  or equal (in absolute value) to endT, updating x
     *  @param x the starting vector. On return, the ending vector.
     *  @param h a two-element array. h[0] is the stepsize to 
     *  try initially, and h[1] is the maximum stepsize. This
     *  routine may use any step size it wants, up to h[1]. Note that
     *  h[0] and h[1] may be negative but both will have the same sign and
     *  abs(h[0]) <= abs(h[1]); this is assured by the calling method.
     *  On return, h[0] is the predicted ideal step size for the next step
     *  and h[1] is the step actually taken
     *  @throws DidNotConvergeException if the solver could not
     *  converge on a solution
     */
    public void solve (Vec x, double[] h)
      throws DidNotConvergeException{

        Vec dxdt = new Vec_array(x.size());
        Vec scale = new Vec_array(x.size());
  
        _dxdt.eval(x, dxdt);
        // scale is from Numerical Recipes, p.718
        // they add an amount TINY to this to prevent division by zero,
        // but the routine maxRatio already watches for this
        for (int i = 0; i < x.size(); i++)
            scale.set(i, Math.abs(x.get(i))+Math.abs(dxdt.get(i)*h[0]));
        solveStep (x, dxdt, h, scale);
        // now allow for multithreading
        Thread.yield();
        // check for interruption, and reraise the
        // interrupt so the calling methods can get
        // interrupted too.
        if (Thread.interrupted()) {
            Thread.currentThread().interrupt();
            return;
        }
    } // solve
     
    /** solve a simple substep of an ODE
     *  @param x the vector to start from
     *  @param dxdt the result of _dxdt.eval(x), passed in
     *  so that solveStep won't call it a second time.
     *  @param h a two-element array. h[0] is the stepsize to 
     *  try initially, and h[1] is the maximum stepsize. This
     *  routine may use any step size it wants, up to h[1]. Note that
     *  h[0] and h[1] may be negative but both will have the same sign and
     *  abs(h[0]) <= abs(h[1]); this is assured by the calling method.
     *  On return, h[0] is the predicted ideal step size for the next step
     *  and h[1] is the step actually taken
     *  @param scale a vector with an estimate of the scale of the elements
     *  of x and dxdt. solveStep should aim for an error of less
     *  than _eps*max(scale), but this is entirely up to the implementation
     *  and scale and _eps can be ignored completely.
     */
    abstract void solveStep (Vec x, Vec dxdt, double[] h, Vec scale)
      throws DidNotConvergeException;
    
    /** utility method to calculate the maximum ratio of elements of two arrays.
     *  @param numer the vector of numerators
     *  @param denom the vector of denominators
     *  @param max the minimum value to return.
     */
    public double maxRatio (Vec numer, Vec denom, double max){
        int i;
        for (i = 0; i < numer.size(); i++){
            if (denom.get(i) != 0){
                max = Math.max (Math.abs(numer.get(i)/denom.get(i)), max);
            } // if
        } // for
        return max;
    } // maxRatio

    /** the name of this ODE solving algorithm, for selection
     *  and display.
     *  @return the name of this ODE algorithm
     */
    public static String name() {
        return "Abstract ODE Solver";
    }

    /** a description of this ODE solving algorithm, for selection
     *  and display.
     *  @return the description of this ODE algorithm
     */
    public static String description(){
        return "Should never be instantiated";
    }
    
    /** an ODE factory that allows changing the algorithm.
     *  See Design Patterns (<i>Factory</i> and <i>Strategy</i>)
     */
    public static class Factory{
        // the list of all possible ODE's
        private Class[] _list = {Euler.class, FluxTolerance.class,
          RungeKutta4.class, EmbeddedRungeKutta.class, MidpointODE.class,
          BulirschStoerODE.class, BaderDeuflhardODE.class, 
          SemiImplicitMidpointODE.class };
        // the currently selected _ode; the one to return from
        // {@link #instance}
        private Class _ode = BulirschStoerODE.class;
        
        public Factory() {}
        public ODE instance (VecFunction dxdt)
          throws NoSuchMethodException, InstantiationException,
          IllegalAccessException,  java.lang.reflect.InvocationTargetException {
            java.lang.reflect.Constructor c = _ode.getConstructor
              (new Class[] {VecFunction.class});
            return (ODE) c.newInstance
              (new Object[] {dxdt});
        } // instance
        
        /** gets the name of the current ODE
         *  @return the name of the currently selected ODE
         */
        public String getODE(){
            try{
                return (String) _ode.
                  getMethod("name", null).invoke (null, null);
            }catch (Exception ex){
                return _ode.getName();
            }
        }
        
        /** sets the current ODE solver
         *  @param name the name of the desired ODE solver
         *  @throws IllegalArgumentException if no such solver exists
         */
        public void setODE (String name){
            if (name == null) throw new IllegalArgumentException();
            for (int i = 0; i < _list.length; i++){
                try{
                    String tryName = (String) _list[i].
                      getMethod("name", null).invoke (null, null);
                    if (name.equals(tryName)){
                      _ode = _list[i];
                      return;
                    } // if
                }catch (Exception ex){
                  // do nothing
                }  // try
            } // for
            throw new IllegalArgumentException(name+" is not a known ODE solver");
        } // setODE
    
        /** the list of names of solvers 
         *  @return an array of String representing the name of each
         *  solving algorithm
         */
        public String[] names() {
            String[] s = new String[_list.length];
            for (int i = 0; i < s.length; i++){
                try{
                    s[i] = (String) 
                      _list[i].getMethod("name", null).invoke (null, null);
                }catch (Exception ex){
                    s[i] = _list[i].toString();
                } // try
            } // for
            return s;
        } // names
        
        /** the list of descriptions of solvers
         *  @return an array of String representing the description of each
         *  solving algorithm
         */
        public String[] descriptions() {
            String[] s = new String[_list.length];
            for (int i = 0; i < s.length; i++){
                try{
                    s[i] = (String) _list[i].
                      getMethod("description", null).invoke (null, null);
                }catch (Exception ex){
                    s[i] = _list[i].toString();
                } // try
            } // for
            return s;
        } // descriptions

    } // ODE.Factory

}  // ODE

// The individual solvers are all package-protected;
// they should only be instantiated
// by ODE.Factory.instance().

