
/** Runge-Kutta 4th order solver; algorithm from Numerical Recipes
 * not adaptive stepsize; takes endT*_eps size steps
 */
package nr.ode;
import nr.*;
class RungeKutta4 extends ODE{

    private double _stepSize;

    public RungeKutta4 (VecFunction dxdt){
        super (dxdt);
    }

    // set _stepsize then solve
    public void solve (Vec x, final double endT)
      throws DidNotConvergeException{
        _stepSize = _eps * endT;
        super.solve (x, endT);
    } // solve

    // set _stepsize then solve
    public void solve (Vec x, double[] h)
      throws DidNotConvergeException{
        _stepSize = _eps * h[1]; // always goes the distance
        super.solve (x, h[1]);
        h[0] = h[1];
    } // solve
   
    void solveStep(Vec x, Vec dxdt, double[] h, Vec scale) throws DidNotConvergeException {
        int i;
        Vec xTemp = new Vec_array(x.size());

        if (_stepSize > 0){
            h[1] = Math.min (h[1], _stepSize);
        }else{
            h[1] = Math.max (h[1], _stepSize);
        }
        double halfT = h[1]*0.5;
 
        // will need to evaluate dxdt at four places: start, halfway (twice)
        // and at the end. The values for the
        // start, we are given in dxdt. The rest we will place in dxdtArray
        Vec[] dxdtArray = new Vec[3];        
        // first step
        for (i=0; i<x.size(); i++){
            xTemp.set (i, x.get(i) + halfT*dxdt.get(i));
        } // for
        // second step
        dxdtArray[0] = _dxdt.eval(xTemp);
        for (i=0; i<x.size(); i++){
            xTemp.set (i, x.get(i) + halfT*dxdtArray[0].get(i));
        } // for
        // third step
        dxdtArray[1] = _dxdt.eval(xTemp);
        for (i=1; i<x.size(); i++) {
            xTemp.set (i, x.get(i) + halfT*dxdtArray[1].get(i));
        } // for
        // fourth step
        dxdtArray[2] = _dxdt.eval(xTemp);
        for (i=0; i<x.size(); i++){
            x.set (i, x.get(i) + h[1] * (dxdt.get(i) + 2.0*dxdtArray[0].get(i)
              + 2.0*dxdtArray[1].get(i) + dxdtArray[2].get(i)) / 6.0);
        } // for
   } // solve

    public static String name() {return "Fourth order Runge Kutta";}
    public static String description(){
        return "Similar to Euler's method (divides each step into " + 
          "1/epsilon substeps) but" +
          "uses a more sophisticated algorithm to estimate the total " + 
          "derivative from " +
          "4 determinations in each step. A good all-purpose algorithm";
    } // description
    
} // RungeKutta4
