/** Implements the semi-implicit modified midpoint algorithm for solving ODE's.
 *  From Numerical Recipes section 16.6,
 *  equation 16.6.30.  Takes a fixed number of substeps
 *  determined as 1/epsilon.
 */
package nr.ode;
import nr.*;

class SemiImplicitMidpointODE extends ODE{
    // true if the Jacobian was not set by the caller.
    // If the caller sets the Jacobian, it must always make sure that it
    // is set before a call to solveStep, or it can setJacobian (null)
    // to indicate that solveStep should calculate it.
    private boolean _needToCalculateJacobian = true;
    private Mat _jacobian = null;
    
    public SemiImplicitMidpointODE(VecFunction dxdt){
        super(dxdt);
    } // constructor
        
    
    // solves a step, assuming _jacobian has been set correctly. 
    void solveStep(Vec x, Vec dxdt, double[] h, Vec scale)
      throws DidNotConvergeException {
        if (_needToCalculateJacobian) _jacobian = _dxdt.jacobian(x);
	int n=x.size();
        // a is the matrix I - h*jacobian
        Mat a = Mat_array.identity(n);
        for (int r = 0; r < n; r++)
            for (int c = 0; c < n; c++)
                a.set (r, c, a.get(r,c) - h[1] * _jacobian.get(r, c));
        LUDecomposition decomp = new LUDecomposition (a);
        if (!decomp.isNonsingular()) throw new DidNotConvergeException();
	Vec delta = new Vec_array(n);
        Vec xTemp = new Vec_array(n);
        int numSteps = (int) (1d/_eps);
        if (numSteps < 2) numSteps = 2;
	double stepSize = h[1] / numSteps;

        // first step. Equation 16.6.32, with xTemp playing the role of y
	for (int i=0; i < n; i++)  delta.set (i, stepSize*dxdt.get(i));
        decomp.solve(delta);
        for (int i=0; i < n; i++)  xTemp.set (i, x.get(i) + delta.get(i));

        // general steps. Equation 16.6.33, with step == k, using
        // dxdt to hold the [h f(y(k)) - delta(k-1)], then solving
        // so it holds the [1-h df/dy][h f - delta] term
	for (int step = 1; step < numSteps; step++) {
            _dxdt.eval(xTemp, dxdt);
            for (int i = 0; i < n; i++){
                dxdt.set(i, stepSize*dxdt.get(i) - delta.get(i));
            } // for i
            decomp.solve(dxdt);
            for (int i = 0; i < n; i++){
                delta.set (i, delta.get(i) + 2d * dxdt.get(i));
                xTemp.set (i, xTemp.get(i) + delta.get(i));
            } // for i
	} // for step
        
        // last step. Equation 16.6.34
        _dxdt.eval (xTemp, dxdt);
	for (int i = 0; i < n; i++){
            delta.set (i, stepSize*dxdt.get(i) - delta.get(i));
        } // for i
        decomp.solve (delta);
        for (int i = 0; i < n; i++){
            x.set (i, xTemp.get(i) + delta.get(i));
        } // for i
    } // solveStep
        
    void setJacobian (Mat j) {
        _jacobian = j;
        _needToCalculateJacobian = (j == null);
    } // setJacobian
    
    public static String name(){
        return "Semi-implicit Midpoint";
    }
    
    public static String description(){
        return "Uses a semi-implicit trapezoidal rule to integrate an ODE. "+
          "Uses a sub-stepsize of "+
          "timeStep/epsilon and evaluates the derivative at each point, "+
          "using the Jacobian of the function at the first point to avoid " +
          "the instability of explicit methods with stiff ODE's. "+
          "A relatively fast but "+
          "inaccurate method. Use epsilon 0.1-.001 for best results.";
    }
} // SemiImplicitMidpointODE