/* CompilerTimer.java
 * Created 5 August 2004
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import tenua.parser.MechanismParser;
import tenua.simulator.Mechanism;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeListenerProxy;
import javax.swing.JComponent;

/** Sets up a timer loop to run the compiler on a regular basis
 *
 *  @author Daniel Wachsstock
 */
public class CompilerTimer extends java.util.Timer{
    private final Tenua _parent;
    private final long _period = 
      Long.parseLong (util.Resources.getString("compiler.period"));

    // Flag that the mechanism has changed
    volatile private boolean _needToCompile = false;

    /** Creates a new CompilerTimer
     *  @param parent the Tenua window that created this
     */
    public CompilerTimer (Tenua parent){
        super (true); // is a daemon
        _parent = parent;
         schedule (new CompilerTask(), 0, _period);
    } // CompilerTimer

    /** called by other threads to alert this one that the mechanism has changed
     */
    public void compile() { _needToCompile = true; }
    
    /** compiles a string with a given SymbolTable into a macro that can be executed by
     *  a mechanism. It translates the variables into get/sut method calls.
     *  @param s the String to compile
     *  @param st the SymbolTable to use
     *  @return a String with s appropriately mangled, to be used
     *  with bsh.Interpreter.eval (String)
     */
    static public String compileMacro (String s, tenua.symbol.SymbolTable st)
      throws tenua.parser.ParseException {
            if (s == null) return "";
            java.io.Reader stream = new java.io.StringReader
              (s);
            MechanismParser parser = new MechanismParser (stream);
            parser.token_source.SwitchTo (parser.IN_SCRIPT);
            return parser.script(st).toString();
    } // compileMacro
    
    /** the task that acutally calls the compiler */
    class CompilerTask extends java.util.TimerTask{
        /** runs the compiler if the mechanism has changed */
        public void run(){
            if (!_needToCompile) return;
            try{
                _needToCompile = false;
                // about to do some dangerous Thread things, which at worst will
                // throw a BadLocationException if the text has changed while we
                // are getting it, but that will be corrected at the
                // next run of the TimerTask
                int len = _parent.mechanismText.getLength();
                String text = _parent.mechanismText.getText(0,len);
                Mechanism mechanism = null;
                _parent.setCompilerErrors("");
                java.io.Reader stream = new java.io.StringReader
                  (text);
                MechanismParser parser = new MechanismParser (stream);
                mechanism = parser.mechanism();
                _parent.setMechanism (mechanism);
            }catch (javax.swing.text.BadLocationException locEx){
                // this will be thrown if len > mechanismText.getLength
                // which is possible if it was edited between execution
                // of those lines. We just have to wait until the next run of
                // the TimerTask to get the full text
                _needToCompile = true;
            }catch (bsh.ParseException parseEx){
                _parent.setCompilerErrors
                  ("Script Error: "+ parseEx.toString());
            }catch (tenua.parser.TokenMgrError tokenEx){
                _parent.setCompilerErrors
                  ("Unexpected character: " + tokenEx.toString());
            }catch (Exception ex){
                _parent.setCompilerErrors (ex.toString());
//                ex.printStackTrace(); // for debugging
            }catch (Throwable t){
                util.ErrorDialog.errorDialog("UnsupportedOperation", t);
//                t.printStackTrace(); // for debugging
            }finally{
                _needToCompile = false;
            } // try
        } // run
        
    } // CompilerTask

} // CompilerTimer
