/*
 * DoubleTextField.java
 *
 * Created on July 7, 2004, 1:36 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import java.awt.Dimension;
import java.beans.PropertyChangeListener;
import javax.swing.Box;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTextField;

/** A JComponent that consists of a JLabel on top of a JTextField.
 *  The text field is constrained to represent a double value.
 *  The value can be accessed via set/get methods.
 *  The name is final and can be accessed with {@link #getName()}.
 *
 *  The value can be tied to a property of a JComponent (with key ==
 *  name) or can be a property of this DoubleTextField itself, also
 *  with key == name.
 *
 *  If it is tied to an outside property, that JComponent should be 
 *  a PropertyChangeListener for itself, so that if this changes the value
 *  the source will know about it.
 *
 * @author  wachdh
 */
public class DoubleTextField extends Box implements PropertyChangeListener {
    private final JLabel _label;
    private final JTextField _text;
    private final javax.swing.JComponent _source;
   
    /** Creates a new instance of DoubleTextField with a given name
     *  and initial value, stored in this
     *  @param name the name
     *  @param value the value
     *  @throws NullPointerException if name is null
     */
    public DoubleTextField (String name, double value){
        this (name, value, null);
    }

    /** Creates a new instance of DoubleTextField with a given name,
     *  with the value stored in a JComponent
     *  @param name the name
     *  @param source the JComponent to use to store the value as a property.
     *  It should be a PropertyChangeListener for itself to catch when this
     *  changes the value. If source == null, use this to store the value
     *  @throws NullPointerException if name is null or source does not have that property
     *  @throws ClassCastException if source's property is not a Number
     */
     public DoubleTextField (String name, JComponent source){
         this (name, ((Number)source.getClientProperty(name)).doubleValue(), source);
     }
     
    /** Creates a new instance of DoubleTextField with a given name
     *  and initial value, stored in a JComponent
     *  @param name the name
     *  @param value the value
     *  @param source the JComponent to use to store the value as a property.
     *  It should be a PropertyChangeListener for itself to catch when this
     *  changes the value. If source == null, use this to store the value
     *  @throws NullPointerException if name is null
     */
    public DoubleTextField (String name, double value, JComponent source){
        super(javax.swing.BoxLayout.Y_AXIS);
        if (name == null) throw new NullPointerException("Null name in "+this.getClass());
        if (source == null) _source = this;
        else _source = source;
        _source.putClientProperty(name, new Double(value));
        super.setName(name);
        _label = new JLabel(name);
        add(_label);
        _text = new JTextField(Double.toString(value));
        _text.setInputVerifier(new DoubleTextField.NumberVerifier(this));
        add(_text);
        _label.setLabelFor(_text);
        if (_source != this) _source.addPropertyChangeListener(name, this);
        // set the maximum size so we keep one-line text fields
        Dimension size = _label.getPreferredSize();
        size.width = Integer.MAX_VALUE;
        size.height *= 2;
        setMaximumSize(size);
    }
    
    /** return the current value
     *  @return the value
     */
    public double getValue() {
        return ((Number)_source.getClientProperty(getName())).doubleValue();
    }
    
    // Box.getName() works just fine; no need to override
    
    /** set the value of the text field
     *  @param value the new value
     */
    public void setValue (double value) {
        double oldValue = getValue();
        if (oldValue == value) return;
        _text.setText(Double.toString(value));
        String name = getName();
        _source.putClientProperty(name, new Double(value));
   }
    
    /** set the displayed name. Ignored by DoubleTextField, since the name
     *  is used to access the value property */
    public void setName (String name){}
    
    public void propertyChange(java.beans.PropertyChangeEvent evt) {
        _text.setText(util.Num2Str.formatNumber(evt.getNewValue()));
        repaint();
    }
    
    /** used to get the actual text in this */
    public String getText() { return _text.getText(); }
    
    class NumberVerifier extends javax.swing.InputVerifier {
        double _d; // used to hold the last value tested; not Thread-safe
        final DoubleTextField _source;
        
        public NumberVerifier (DoubleTextField source) { _source = source; }
        
        public boolean verify(JComponent input) {
            try{
                _d = Double.parseDouble(_source.getText());
                return true;
            }catch (NumberFormatException ex){
                _d = _source.getValue(); // reset the value to the original
                return false;
            } // try
        } // verify

        public boolean shouldYieldFocus(JComponent input){
            verify(input);
            _source.setValue(_d); // always change the value, either to reset or to the new one
            return true; // always yield focus
        } // shouldYieldFocus
    } // NumberVerifier
} // DoubleTextField
