/* EditorPanel
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import java.awt.Color;
import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.print.PageFormat;
import javax.swing.*;
import javax.swing.event.DocumentListener;
import javax.swing.text.DefaultEditorKit;
import javax.swing.text.Document;

/** creates a panel that contains a simple text editor on top and a
 *  text area on the bottom for error messages
 *
 *  @author Daniel Wachsstock
 */
public class EditorPanel extends JPanel
  implements DocumentListener{
    private final Tenua _parent;
    private final JTextArea _editor;
    private final JTextArea _errors;
    private final JLabel _errorHeader;
    private final util.StringFiler _file;
    private final UndoHandler _undo;
    private boolean _changed = false; // for tracking whether we need to save

    /** create a new EditorPanel
     *  @param parent the Tenua window that created this */
    public EditorPanel (Tenua parent){
        super();
        _parent = parent;
        getActionMap().setParent(_parent.getActionMap());
        
        setLayout(new BorderLayout());
        _editor = new JTextArea (_parent.mechanismText);
        _editor.setFont (new Font ("monospaced", Font.PLAIN, 12));
        _errors = new JTextArea (_parent.mechanismCompilerErrors);
        _errors.setEditable (false);
        _errors.setForeground (Color.RED);
        _errors.setFont(_editor.getFont());
        _errorHeader = new JLabel (util.Resources.getString("CompilerErrorHeader"));
        _errorHeader.setLabelFor(_errors);
        JPanel errorPanel = new JPanel ();
        errorPanel.setLayout(new BoxLayout (errorPanel, BoxLayout.PAGE_AXIS));
        errorPanel.add (_errorHeader);
        errorPanel.add (new JScrollPane(_errors));
        errorPanel.setMinimumSize (new java.awt.Dimension(0, 60));
        JSplitPane splitPane = new JSplitPane
         (JSplitPane.VERTICAL_SPLIT, new JScrollPane(_editor), errorPanel);
        splitPane.setOneTouchExpandable (true);
        splitPane.setResizeWeight (1.0); // expand the editor, not the error panel
        add(splitPane);
        
        loadSpecificActions();

        _parent.mechanismText.addDocumentListener(this);
        _parent.mechanismCompilerErrors.addDocumentListener(this);
        _file = new util.StringFiler(_editor.getText()); // no name to start
        _undo = new UndoHandler(); // needs to be initialized with the other final fields set
        
        // there's an odd bug such that scrolling fails from within a document listener
        // so we catch changes in the error listing here, and automatically scroll to the top
        _errors.addComponentListener(new java.awt.event.ComponentAdapter(){
            public void componentResized(java.awt.event.ComponentEvent e){
              _errors.scrollRectToVisible(new java.awt.Rectangle());
            } // componentResized
        }); // addComponentListener
        
        this.setVisible(true);
    } // constructor
    
    /** do nothing for attributes changing */
    public void changedUpdate(javax.swing.event.DocumentEvent e) {}
    
    /** respond to document text inserted events */
    public void insertUpdate(javax.swing.event.DocumentEvent e) {
        documentChanged (e.getDocument());
    } // insertUpdate
    
    /** respond to document text removed events */
    public void removeUpdate(javax.swing.event.DocumentEvent e) {
        documentChanged (e.getDocument());
    } // removeUpdate
    
    /** deal with documents changing: if the main document, 
     *  set the flag so it can be saved;
     *  if the error document, mark it in red */
    public void documentChanged (Document d){
        if (d == _parent.mechanismText){
            _changed = true;
        }else if (d == _parent.mechanismCompilerErrors){
            // flag errors
            _errorHeader.setForeground(d.getLength() == 0 ? Color.BLACK : Color.RED);
       } // if
    } // documentChanged
    
    /** Sets the focus of the window to the editor text area */
    public void acceptFocus(){
        _editor.requestFocusInWindow();
    }
    
    // adds appropriate actions to this action map
    private void loadSpecificActions(){
        final ActionMap map = this.getActionMap();

        // LOAD
        map.put(ActionType.LOAD, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                String name = _file.getFilename(); // old name
                _file.setFilename(null); // make sure we ask for a file name
                String text = _file.load();
                if (text != null) {
                    // we set the name of the window to the file name
                    ((JFrame)SwingUtilities.windowForComponent(_parent))
                      .setTitle(_file.getShortName());
                    _editor.setText(text);
                    _changed = false;  // documentChanged will have been called as a side
                      // effect of the previous line, but we undo now
                }else{
                    // a cancel; reset the file name
                    _file.setFilename(name);
                } // if
            } // actionPerformed
        }); // map.put

        // SAVE
        map.put(ActionType.SAVE, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                String name = _file.getFilename();
                _file.save(_editor.getText());
                if (_file.getFilename() != null) {
                    // we set the name of the window to the file name
                    ((JFrame)SwingUtilities.windowForComponent(_parent))
                      .setTitle(_file.getShortName());
                    _changed = false;
                } // if
            } // actionPerformed
        }); // map.put

        // SAVE AS
        map.put(ActionType.SAVE_AS, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                String name = _file.getFilename();
                _file.saveAs(_editor.getText());
                if (_file.getFilename() != null) {
                    // we set the name of the window to the file name
                    ((JFrame)SwingUtilities.windowForComponent(_parent))
                      .setTitle(_file.getShortName());
                    _changed = false;
                } // if
            } // actionPerformed
        }); // map.put

        
        // CLOSE
        map.put(ActionType.CLOSE, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                if (_changed){
                    int result =
                      JOptionPane.showConfirmDialog(_parent, 
                      util.Resources.getString ("ShouldSavePrompt"),
                      util.Resources.getString ("ShouldSaveTitle"),
                      JOptionPane.YES_NO_CANCEL_OPTION);
                    if (result == JOptionPane.YES_OPTION){
                        map.get(ActionType.SAVE).actionPerformed(e);
                    }else if (result == JOptionPane.CANCEL_OPTION){
                        putValue ("closeCancelled", Boolean.TRUE);
                    }
                } // if
            } // actionPerformed
        }); // map.put
        
        // PRINT
        map.put (ActionType.PRINT, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                PageFormat format = TenuaAction.getFormat(_parent);
                StringBuffer text = util.StringPrinter.untabify
                  (_editor.getText(), 8, false);
                util.StringPrinter job = new util.StringPrinter
                  (text.toString(), _editor.getFont(), format);
                String title = ((JFrame)SwingUtilities.windowForComponent
                  (_parent)).getTitle();
                job.setTitle (title);
                job.start();
            } // actionPerformed
        }); // map.put

        // UNDO
        map.put (ActionType.UNDO, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                try{
                    _undo.undo();
                    _undo.update();
                }catch (javax.swing.undo.CannotUndoException ex){
                    util.ErrorDialog.errorDialog("CannotUndo", ex);
                } // try
            } // actionPerformed
        }); // map.put

        // REDO
        map.put (ActionType.REDO, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                try{
                    _undo.redo();
                    _undo.update();
                }catch (javax.swing.undo.CannotRedoException ex){
                    util.ErrorDialog.errorDialog("CannotRedo", ex);
                } // try
            } // actionPerformed
        }); // map.put
        
        // CUT, COPY, PASTE, SELECT ALL
        // the editor kit actions are designed to be used from the keyboard,
        // so if used from a menu or toolbar the editor loses the focus. We have to change this
        class EditorAction extends AbstractAction{
            private final Action _a;
            public EditorAction(Action a) {_a = a;}
            public void actionPerformed (ActionEvent e){
                _a.actionPerformed(e);
                _editor.requestFocusInWindow();
            }
        } // EditorAction
        // grab the actions we want from the editor. There ought to be an easier way
        Action[] actions = _editor.getActions();
        for (int i = 0; i < actions.length; i++){     
            if (DefaultEditorKit.cutAction.equals(actions[i].getValue(Action.NAME))){
                map.put (ActionType.CUT, new EditorAction(actions[i]));
            } // if
            if (DefaultEditorKit.copyAction.equals(actions[i].getValue(Action.NAME))){
                map.put (ActionType.COPY, new EditorAction(actions[i]));
            } // if
            if (DefaultEditorKit.pasteAction.equals(actions[i].getValue(Action.NAME))){
                map.put (ActionType.PASTE, new EditorAction(actions[i]));
            } // if
            if (DefaultEditorKit.selectAllAction.equals(actions[i].getValue(Action.NAME))){
                map.put (ActionType.SELECT_ALL, new EditorAction(actions[i]));
            } // if
        } // for
                   
        // CLEAR
        map.put (ActionType.CLEAR, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                int start = _editor.getCaret().getDot();
                int end = _editor.getCaret().getMark();
                int len;
                if (start > end){
                    len = start-end;
                    start = end;
                }else{
                    len = end-start;
                } // if
                try{
                    _parent.mechanismText.remove (start, len);
                }catch (Exception ex){
                    // do nothing if BadLocationException is thrown
                } // try
            } // actionPerformed
        }); // map.put
                
    } // loadSpecificActions
    
    // class to handle undo
    class UndoHandler extends javax.swing.undo.UndoManager{
        private final Action undo =
          getActionMap().get(ActionType.UNDO.name());
        private Action redo =
          getActionMap().get(ActionType.REDO.name());;
        UndoHandler(){
            _parent.mechanismText.addUndoableEditListener (this);
            update();
        } // constructor
        
        public void undoableEditHappened (javax.swing.event.UndoableEditEvent e){
            super.undoableEditHappened(e);
            update();
         } // undoableEditHappened
        
        public void update(){
            if (canUndo()){
                undo.setEnabled(true);
                undo.putValue(Action.NAME, _undo.getUndoPresentationName());
            }else{
                undo.setEnabled(false);
            } // if
            if (canRedo()){
                redo.setEnabled(true);
                redo.putValue(Action.NAME, _undo.getRedoPresentationName());
            }else{
                redo.setEnabled(false);
            } // if
        } // update
    } // UndoHandler
} // EditorPanel