/*
 * GraphPanel.java
 *
 * Created on March 30, 2004, 6:52 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import javax.swing.JPanel;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.event.*;
import java.awt.geom.Rectangle2D;
import tenua.simulator.DataTableImpl;
import util.Num2Str;

/** plots a graph
 *  based on a DataModel
 * @author  wachdh
 */
public class GraphPanel extends JPanel implements TableModelListener, PropertyChangeListener {
    private final Tenua _parent;
    private final DataTableImpl _data;
    private double _tmin = 0d;
    private double _tmax = 10d;
    private double _xmin = 0d;
    private double _xmax = 10d;
    private boolean _adjustAxes = true;
    private Rectangle _dim;
    private final static int TICSIZE = 5;
    private final static int VERTICAL = 0;
    private final static int HORIZONTAL = 1;   
    
    /** key for the maximum of the time axis property */
    public static final String TMAX = util.Resources.getString("Axes.tmax");
    /** key for the minimum of the time axis property */
    public static final String TMIN = util.Resources.getString("Axes.tmin");
    /** key for the maximum of the value axis property */
    public static final String XMAX = util.Resources.getString("Axes.xmax");
    /** key for the minimum of the value axis property */
    public static final String XMIN = util.Resources.getString("Axes.xmin");
    /** key for the boolean to automatically adjust axes to fit the data */
    public static final String ADJUST_AXES = util.Resources.getString("Axes.adjustAxes");
    
    /** Creates a new instance of GraphPanel, tied to a model.
     *  @param model the DataModel to plot.
     *  Plots x vs. time, so
     *  what would normally be called the x-axis is here called the t-axis (for
     *  time), and what would normally be called the y-axis is here called the
     *  x-axis
     */
    public GraphPanel(Tenua parent){
        _parent = parent;
        _data = parent.data;
        _data.addTableModelListener(this);
        setBackground (Color.white);
        setFont (new Font("SansSerif", Font.PLAIN, 12));
        putClientProperty (TMIN, new Double(_tmin));
        putClientProperty (TMAX, new Double(_tmax));
        putClientProperty (XMIN, new Double(_xmin));
        putClientProperty (XMAX, new Double(_xmax));
        putClientProperty (ADJUST_AXES, new Boolean(_adjustAxes));
        addPropertyChangeListener(this);
    } // constructor
    
    /** repaints the entire graph when the data changes */
    public void tableChanged (TableModelEvent e){
        repaint(); // we need to draw everything again
    } // tableChanged
    
    /** set the minimum time displayed
     *  @param tmin the new value
     *  @param setProperty true if this function should call firePropertyChanged;
     *  false if this is being called from a property change event, so calling
     *  firePropertyChanged would be redundant.
     */
    public void setTMin (double tmin, boolean setProperty){
        if (tmin == _tmin) return;
        if (tmin > _tmax){
            // flip them if they are reversed
            setTMin (_tmax, true);
            setTMax (tmin, true);
            return;
        }else if (tmin == _tmax){
            // not allowed to have zero-sized axes; guess at an appropriate scale
            setTMax (tmin+10d, true);
        }
        if (setProperty) putClientProperty (TMIN, new Double(tmin));
        _tmin = tmin;
        repaint();
    } // setTMin
    
    /** set the maximum time displayed
     *  @param tmax the new value
     *  @param setProperty true if this function should call firePropertyChanged;
     *  false if this is being called from a property change event, so calling
     *  firePropertyChanged would be redundant.
     */
    public void setTMax (double tmax, boolean setProperty){
        if (tmax == _tmax) return;
        if (tmax < _tmin){
            // flip them
            setTMax (_tmin, true);
            setTMin (tmax, true);
            return;
        }else if (tmax == _tmin) {
            // not allowed to have zero-sized axes; guess at an appropriate scale
            setTMin (tmax-10d, true);            
        }
        if (setProperty) putClientProperty (TMAX, new Double(tmax));
        _tmax = tmax;
        repaint();
    } // setTMax
    
    /** set the minimum x (vertical) scale
     *  @param xmin the new value
     *  @param setProperty true if this function should call firePropertyChanged;
     *  false if this is being called from a property change event, so calling
     *  firePropertyChanged would be redundant.
     */
    public void setXMin (double xmin, boolean setProperty){
        if (xmin == _xmin) return;
        if (xmin > _xmax){
            // flip them if they are reversed
            setXMin (_xmax, true);
            setXMax (xmin, true);
            return;
        }else if (xmin == _xmax){
            // not allowed to have zero-sized axes; guess at an appropriate scale
            setXMax (xmin+10d, true);
        }
        if (setProperty) putClientProperty (XMIN, new Double(xmin));
        _xmin = xmin;
        repaint();
    } // setXMin
    
    /** set the maximum x (vertical) scale
     *  @param xmax the new value
     *  @param setProperty true if this function should call firePropertyChanged;
     *  false if this is being called from a property change event, so calling
     *  firePropertyChanged would be redundant.
     */
   public void setXMax (double xmax, boolean setProperty){
        if (xmax == _xmax) return;
        if (xmax < _xmin){
            // flip them
            setXMax (_xmin, true);
            setXMin (xmax, true);
            return;
        }else if (xmax == _xmin) {
            // not allowed to have zero-sized axes; guess at an appropriate scale
            setXMin (xmax-10d, true);            
        }
        if (setProperty) putClientProperty (XMAX, new Double(xmax));
        _xmax = xmax;
        repaint();
    } // setTMin
    
    /** determines whether to automatically expand the scale of the axes
     *  if the data will not fit.
     *  @param adjustAxes true to turn on automatic axis adjustment
     *  @param setProperty true if this function should call firePropertyChanged;
     *  false if this is being called from a property change event, so calling
     *  firePropertyChanged would be redundant.
     */
    public void setAdjustAxes (boolean adjustAxes, boolean setProperty){
        if (adjustAxes == _adjustAxes) return;
        _adjustAxes = adjustAxes;
        if (setProperty) putClientProperty (ADJUST_AXES, new Boolean(adjustAxes));
        if (adjustAxes && !_adjustAxes) repaint();
    } // setAdjustAxes
    
    /** @return the minimum time displayed */
    public double getTMin() { return _tmin; }

    /** @return the maximum time displayed */
    public double getTMax() { return _tmax; }

    /** @return the minimum x (vertical) scale */
    public double getXMin() { return _xmin; }

    /** @return the maximum x (vertical) scale */
    public double getXMax() { return _xmax; }

    /** @return whether the axes are automatically adjusted */
    public boolean getAdjustAxes() { return _adjustAxes; }
       
    /** erases the background, draws the axes and all the curves */
    public void paint (Graphics g){
        _dim = this.getBounds(_dim);
        // subtract the size of labels. Leave room for axis labels at left and bottom
        // and curve labels at right and top. The pattern chosen here should be
        // large enough for any number, but may be too small for the curve labels
        Rectangle2D labelRect = getFontMetrics(getFont()).
          getStringBounds(Num2Str.MAX_STRING, g);
        _dim.x += labelRect.getWidth();
        _dim.y += labelRect.getHeight();
        _dim.width -= 2*labelRect.getWidth();
        _dim.height -= 2*labelRect.getHeight();
        try{
            paintBackground(g);
            paintAxes(g);
            paintCurves(g);
        }catch (Exception ex) { /* do nothing */ }
    } // paint
   
    private void paintBackground (Graphics g){
        g.setColor(Color.white);
        Rectangle rect = this.getBounds();
        g.fillRect (rect.x, rect.y, rect.width, rect.height);
    } // paintBackground
    
    private void paintAxes (Graphics g){
        g.setColor(Color.black);

        // t axis (horizontal)
        double x = 0; // x intercept of the t axis
        if (_xmin > 0) x = _xmin;
        if (_xmax < 0) x = _xmax;
        g.drawLine (scaleT(_tmin), scaleX(x), scaleT(_tmax), scaleX(x));

        // x axis (vertical)
        double t = 0; // t intercept of the tx axis
        if (_tmin > 0) t = _tmin;
        if (_tmax < 0) t = _tmax;
        g.drawLine (scaleT(t), scaleX(_xmin), scaleT(t), scaleX(_xmax));

        // draw tics
        double ticSpacing =
          powerOf10 (Math.max (Math.abs(t-_tmin), Math.abs(t-_tmax)));
        for (double ticTotal = t + ticSpacing; ticTotal < _tmax; ticTotal += ticSpacing){
            drawTic (ticTotal, x, VERTICAL, g);
        } // for
        for (double ticTotal = t - ticSpacing; ticTotal > _tmin; ticTotal -= ticSpacing){
            drawTic (ticTotal, x, VERTICAL, g);
        } // for
        
        ticSpacing = powerOf10(Math.max(Math.abs(x-_xmin), Math.abs(x-_xmax)));
        for (double ticTotal = x + ticSpacing; ticTotal < _xmax; ticTotal += ticSpacing){
            drawTic (t, ticTotal, HORIZONTAL, g);
        } // for
        for (double ticTotal = x - ticSpacing; ticTotal > _xmin; ticTotal -= ticSpacing){
            drawTic (t, ticTotal, HORIZONTAL, g);
        } // for
    } // paintAxes

    /** draws a tic mark centered at h,v with the given orientation */
    private void drawTic (double h, double v, int orientation, Graphics g){
        String label = "";
        Rectangle2D labelRect = null;
        g.setFont(getFont());
        switch (orientation){
            case VERTICAL:
                label = Num2Str.formatNumber(h);
                labelRect = g.getFontMetrics().getStringBounds(label, g);
                // move the label to just under the axis
                offset (labelRect, scaleT(h), scaleX(v) + (int)labelRect.getHeight()/2 - 1);
                g.drawLine (scaleT(h), scaleX(v)-TICSIZE, scaleT(h), scaleX(v));
                break;
            case HORIZONTAL:
                label = Num2Str.formatNumber(v);
                labelRect = g.getFontMetrics().getStringBounds(label, g);
                // move the label to just left of the axis
                offset (labelRect, scaleT(h)- TICSIZE - (int)labelRect.getWidth()/2, scaleX(v)-1);
                g.drawLine (scaleT(h), scaleX(v), scaleT(h)+TICSIZE, scaleX(v));
                break;
        } // switch
        g.drawString(label, (int) labelRect.getX(), (int) labelRect.getMaxY());
    } // drawTic
    
    /** moves rect to center at h,v */
    private void offset (Rectangle2D rect, int h, int v){
        rect.setRect(rect.getX() - rect.getCenterX() + h, rect.getY() - rect.getCenterY() + v,
          rect.getWidth(), rect.getHeight());
    } // offset
    
    private void paintCurves(Graphics g) throws InterruptedException {
        // notation is a bit confusing since Data table uses (x,y) and this
        // uses (t, x).
        g.setColor (Color.black);
        double lastX=0, lastT=0;
        for (int c = 0; c < _data.getColumnCount(); c++){
            if (_data.isHidden(c) && !_parent.getShowHidden()) continue;
            int which = 0;
            g.setColor (Color.getHSBColor(c*1f/_data.getColumnCount(),1,1));
            for (int r = 0; r < _data.getRowCount(); r++){
                double t = _data.getXAt(r);
                if (_adjustAxes && t >= _tmax) setTMax (t<0 ? 0.9*t : 1.1*t, true);
                if (_adjustAxes && t <= _tmin) setTMin (t<0 ? 1.1*t : 0.9*t, true);
                double x = _data.getYAt (r,c);
                if (!Double.isNaN(x)){
                    if (_adjustAxes && x >= _xmax) setXMax (x<0 ? 0.9*x : 1.1*x, true);
                    if (_adjustAxes && x <= _xmin) setXMin (x<0 ? 1.1*x : 0.9*x, true);
                    if (which > 0){
                        g.drawLine(scaleT(lastT), scaleX(lastX), scaleT(t), scaleX(x));
                    } // if
                    lastT = t; lastX = x;
                    ++which;
                } // if
            } // for r
            // if any points were plotted, add the label
            if (which == 1){ // only one point; make an x
                int t = scaleT(lastT);
                int x = scaleX(lastX);
                g.drawLine (t-1, x-1, t+1, x+1);
                g.drawLine (t-1, x+1, t+1, x-1);
            } // if
            if (which > 0) 
              g.drawString(_data.getColumnName(c), scaleT(lastT)+2, scaleX(lastX));
        } // for c       
    } // paintCurves
    
    /** Sets the axes to show all the data.
     *  Adjusts tmin, tmax, xmin and xmax such that all the data are visible,
     *  the (0,0) point is visible, and the largest data value is just 10% smaller
     *  than the extent of the graph
     */
    public void optimizeAxes() throws InterruptedException {
        double tmin = 0;
        double tmax = 0;
        double xmin = 0;
        double xmax = 0;
        for (int c = 0; c < _data.getColumnCount(); c++){
            for (int r = 0; r < _data.getRowCount(); r++){
                double t = _data.getXAt(r);
                if (t < tmin) tmin = t;
                if (t > tmax) tmax = t;
                double x = _data.getYAt (r,c);
                if (!Double.isNaN(x)){
                    if (x < xmin) xmin = x;
                    if (x > xmax) xmax = x;
               } // if
            } // for r
        } // for c
        setTMax (1.1 * tmax, true);
        setTMin (1.1 * tmin, true);
        setXMax (1.1 * xmax, true);
        setXMin (1.1 * xmin, true);
    } // optimizeAxes
    
    private int scaleT (double t){
        return _dim.x + (int) ((t - _tmin)*_dim.width/(_tmax-_tmin));
    } // scaleT
    
    private int scaleX (double x){ // need to scale and flip
        return _dim.y + (int) ((_xmax-x)*_dim.height/(_xmax-_xmin));
    } // scaleX
    
    /** returns the highest power of 10 less than abs(d) 
     *  except if d is higher than but close to a power of 10, then it uses the next
     *  lower power
     */
    static final private double powerOf10 (double d){
        // one could, I suppose, convert to bits and mask off the exponent, but
        // that seems too error-prone
        d = Math.abs(d);
        double result = Math.pow(10, Math.floor(Math.log(d)/Num2Str.LOG10));
        if (2*result > d) result /= 10;
        return result;
    } // powerOf10
    
    public void propertyChange(PropertyChangeEvent evt) {
        String name = evt.getPropertyName();
        if (TMIN.equals(name)){
            setTMin(((Number)evt.getNewValue()).doubleValue(), false);
        } else if (TMAX.equals(name)){
            setTMax(((Number)evt.getNewValue()).doubleValue(), false);
        } else if (XMIN.equals(name)){
            setXMin(((Number)evt.getNewValue()).doubleValue(), false);
        } else if (XMAX.equals(name)){
            setXMax(((Number)evt.getNewValue()).doubleValue(), false);
        }else if (ADJUST_AXES.equals(name)){
            setAdjustAxes(((Boolean)evt.getNewValue()).booleanValue(), false);
        }
    } // propertyChange
    

} // GraphPanel
