/* NotePanel
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.print.PageFormat;
import javax.swing.*;
import javax.swing.text.DefaultEditorKit;
import javax.swing.text.Document;

/** creates a panel that contains a simple text editor on top and a
 *  text area on the bottom for error messages
 *
 *  @author Daniel Wachsstock
 */
public class NotePanel extends JPanel {
    private final Tenua _parent;
    private final JTextArea _editor;
    private final UndoHandler _undo;

    /** create a new NotePanel
     *  @param parent the Tenua window that created this */
    public NotePanel (Tenua parent){
        super();
        _parent = parent;
        getActionMap().setParent(_parent.getActionMap());
        
        setLayout(new java.awt.BorderLayout());
        _editor = new JTextArea (_parent.noteText);
        _editor.setFont (new Font ("monospaced", Font.PLAIN, 12));
        add(new JScrollPane(_editor));
        
        loadSpecificActions();
        _undo = new UndoHandler(); // needs to be initialized with the other final fields set
        
        this.setVisible(true);
    } // constructor
    
    
    /** Sets the focus of the window to the editor text area */
    public void acceptFocus(){
        _editor.requestFocusInWindow();
    }
    
    // adds appropriate actions to this action map
    private void loadSpecificActions(){
        final ActionMap map = this.getActionMap();


        // SAVE
        map.put(ActionType.SAVE, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                util.StringFiler.save(_editor.getText(), null);
            } // actionPerformed
        }); // map.put

        // SAVE AS
        map.put(ActionType.SAVE_AS, map.get (ActionType.SAVE));
        
        
        // PRINT
        map.put (ActionType.PRINT, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                PageFormat format = TenuaAction.getFormat(_parent);
                StringBuffer text = util.StringPrinter.untabify(_editor.getText(), 8, false);
                util.StringPrinter job = new util.StringPrinter
                  (text.toString(), _editor.getFont(), format);
                String title = ((JFrame)SwingUtilities.windowForComponent
                  (_parent)).getTitle();
                job.setTitle (title);
                job.start();
            } // actionPerformed
        }); // map.put

        // UNDO
        map.put (ActionType.UNDO, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                try{
                    _undo.undo();
                    _undo.update();
                }catch (javax.swing.undo.CannotUndoException ex){
                    util.ErrorDialog.errorDialog("CannotUndo", ex);
                } // try
            } // actionPerformed
        }); // map.put

        // REDO
        map.put (ActionType.REDO, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                try{
                    _undo.redo();
                    _undo.update();
                }catch (javax.swing.undo.CannotRedoException ex){
                    util.ErrorDialog.errorDialog("CannotRedo", ex);
                } // try
            } // actionPerformed
        }); // map.put
        
        // CUT, COPY, PASTE, SELECT ALL
        // the editor kit actions are designed to be used from the keyboard,
        // so if used from a menu or toolbar the editor loses the focus. We have to change this
        class EditorAction extends AbstractAction{
            private final Action _a;
            public EditorAction(Action a) {_a = a;}
            public void actionPerformed (ActionEvent e){
                _a.actionPerformed(e);
                _editor.requestFocusInWindow();
            }
        } // EditorAction
        // grab the actions we want from the editor. There ought to be an easier way
        Action[] actions = _editor.getActions();
        for (int i = 0; i < actions.length; i++){     
            if (DefaultEditorKit.cutAction.equals(actions[i].getValue(Action.NAME))){
                map.put (ActionType.CUT, new EditorAction(actions[i]));
            } // if
            if (DefaultEditorKit.copyAction.equals(actions[i].getValue(Action.NAME))){
                map.put (ActionType.COPY, new EditorAction(actions[i]));
            } // if
            if (DefaultEditorKit.pasteAction.equals(actions[i].getValue(Action.NAME))){
                map.put (ActionType.PASTE, new EditorAction(actions[i]));
            } // if
            if (DefaultEditorKit.selectAllAction.equals(actions[i].getValue(Action.NAME))){
                map.put (ActionType.SELECT_ALL, new EditorAction(actions[i]));
            } // if
        } // for
                   
        // CLEAR
        map.put (ActionType.CLEAR, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                int start = _editor.getCaret().getDot();
                int end = _editor.getCaret().getMark();
                int len;
                if (start > end){
                    len = start-end;
                    start = end;
                }else{
                    len = end-start;
                } // if
                try{
                    _parent.noteText.remove (start, len);
                }catch (Exception ex){
                    // do nothing if BadLocationException is thrown
                } // try
            } // actionPerformed
        }); // map.put
                
    } // loadSpecificActions
    
    // class to handle undo
    class UndoHandler extends javax.swing.undo.UndoManager{
        private final Action undo =
          getActionMap().get(ActionType.UNDO.name());
        private Action redo =
          getActionMap().get(ActionType.REDO.name());;
        UndoHandler(){
            _parent.noteText.addUndoableEditListener (this);
            update();
        } // constructor
        
        public void undoableEditHappened (javax.swing.event.UndoableEditEvent e){
            super.undoableEditHappened(e);
            update();
         } // undoableEditHappened
        
        public void update(){
            if (canUndo()){
                undo.setEnabled(true);
                undo.putValue(Action.NAME, _undo.getUndoPresentationName());
            }else{
                undo.setEnabled(false);
            } // if
            if (canRedo()){
                redo.setEnabled(true);
                redo.putValue(Action.NAME, _undo.getRedoPresentationName());
            }else{
                redo.setEnabled(false);
            } // if
        } // update
    } // UndoHandler
} // EditorPanel