/*
 * TablePanel.java
 *
 * Created on April 1, 2004, 7:54 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import java.text.NumberFormat;
import tenua.simulator.*;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.DataFlavor;
import java.awt.event.MouseAdapter;
import javax.swing.DefaultCellEditor;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;
import util.DoubleBean;

/** A Panel that includes a 2-D table of data
 *
 * @author  Daniel Wachsstock
 */
public class TablePanel extends JPanel{
    
    private final Tenua _parent;
    private final JTable _table;
    private final JTable _rowHeaders;
    
    /** Creates a new instance of TablePanel */
    public TablePanel(Tenua parent) {
        super ();
        _parent = parent;
        setLayout (new BorderLayout());
        
        _table = new JTable(_parent.data);
        _table.setAutoResizeMode(_table.AUTO_RESIZE_OFF);
        _table.setRowSelectionAllowed(false);
        _table.setColumnSelectionAllowed (true);
        _table.setDefaultRenderer(Double.class, DoubleBean.cellRenderer());
        _table.setDefaultEditor(Double.class, DoubleBean.cellEditor());
        final javax.swing.table.JTableHeader header = _table.getTableHeader();
        header.setDefaultRenderer(new DefaultTableCellRenderer(){
            public Component getTableCellRendererComponent
              (JTable t, Object o, boolean b1, boolean b2,int r, int c){
                // set the colors to match the graph
                setHorizontalAlignment(CENTER);
                setBorder(javax.swing.UIManager.getBorder("TableHeader.cellBorder"));
                setBackground(Color.LIGHT_GRAY);
                setForeground(Color.getHSBColor
                  (c*1f/_parent.data.getColumnCount(),1,1));
                return super.getTableCellRendererComponent (t,o,b1,b2,r,c);
            } // getTableCellRendererComponent        
        }); // new cell renderer
        header.addMouseListener(new MouseAdapter(){
            // add a double-clock to edit for the column header
            public void mouseClicked (java.awt.event.MouseEvent e){
                if (e.getClickCount() != 2) return;
                int col = header.getColumnModel().getColumnIndexAtX(e.getX());
                if (col == -1) return;
                String prompt = util.Resources.getString ("ChangeColumnDialogPrompt");
                String str = JOptionPane.showInputDialog
                  (_parent, prompt, _parent.data.getColumnName(col));
                if (str != null) _parent.data.setColumnName (col, str);
            } // mouseClicked
        }); // addMouseListener

        // if I just setRowHeaderView(_rowHeaders) it won't stay the right size
        // so I have to put it in a panel that I then set. An ugly kludge; I don't know
        // enough about swing to know if it's a bug
        _rowHeaders = new JTable(_parent.data.getRowHeaderModel());
        _rowHeaders.setAutoResizeMode(_table.AUTO_RESIZE_OFF);
        _rowHeaders.setRowSelectionAllowed(false);
        _rowHeaders.setColumnSelectionAllowed (false);
        _rowHeaders.addMouseListener (new MouseAdapter(){
            // if the user click in the row header, unselect the rest of the table
            public void mouseClicked (java.awt.event.MouseEvent e){
                _table.clearSelection();
            } // mouseClicked
        }); // addMouseListener
        JPanel headerPanel = new JPanel(new BorderLayout());
        headerPanel.add(_rowHeaders);

        JScrollPane pane = new JScrollPane(_table);
        _rowHeaders.setBackground (pane.getViewport().getBackground());
        pane.setRowHeaderView(headerPanel);
        
        String time = util.Resources.getString ("DataTable.Time");
        JLabel timeLabel = new JLabel(time, SwingConstants.CENTER);
        timeLabel.setBorder (javax.swing.UIManager.getBorder("TableHeader.cellBorder"));
        pane.setCorner (pane.UPPER_LEFT_CORNER, timeLabel);

        add (pane);
        
        loadSpecificActions();
    } // constructor
   
    // adds appropriate actions to this action map
    private void loadSpecificActions(){
        final javax.swing.ActionMap map = this.getActionMap();

        // LOAD
        map.put(ActionType.LOAD, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                getDataFromString(util.StringFiler.load(null));
            } // actionPerformed
        }); // map.put

        // SAVE
        // saves the selected columns if any are selected, or the whole table otherwise
        map.put(ActionType.SAVE, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                int[] cols = _table.getSelectedColumns();
                convertColumnIndicesToModel(cols);
                String text = cols.length > 0 ?
                  _parent.data.toString(cols) :
                  _parent.data.toString();
                util.StringFiler.save(text, null);
            } // actionPerformed
        }); // map.put

        // SAVE_AS
        // same as SAVE for tables (no default file name)
        map.put(ActionType.SAVE_AS, map.get (ActionType.SAVE));
        
        // PRINT
        map.put (ActionType.PRINT, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                StringBuffer text = util.StringPrinter.untabify
                  (_parent.data.toString(), 10, true);
                util.StringPrinter job = new util.StringPrinter
                  (text.toString(), TenuaAction.getFormat(_parent));
                String title = ((JFrame)SwingUtilities.windowForComponent
                  (_parent)).getTitle();
                job.setTitle (title + " " +
                   util.Resources.getString("TableTab").split(";")[0]);
                job.start();
            } // actionPerformed
        }); // map.put

        // CUT
        map.put(ActionType.CUT, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                // just do a COPY then a CLEAR
                map.get(ActionType.COPY).actionPerformed (e);
                map.get(ActionType.CLEAR).actionPerformed(e);
            } // actionPerformed
        }); // map.put

        // COPY
        map.put(ActionType.COPY, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                int[] cols = _table.getSelectedColumns();
                convertColumnIndicesToModel(cols);
                StringSelection text = new StringSelection
                  (_parent.data.toString(cols));
                TablePanel.this.getToolkit().getSystemClipboard()
                  .setContents(text, text);
            } // actionPerformed
        }); // map.put

        // PASTE
        map.put(ActionType.PASTE, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                  java.awt.datatransfer.Transferable t = TablePanel.this.getToolkit().
                    getSystemClipboard().getContents(this);
                  try{
                      TablePanel.this.getDataFromString ((String)
                        t.getTransferData(DataFlavor.stringFlavor));
                  }catch (Exception ex) { /* do nothing */ }
            } // actionPerformed
        }); // map.put

        // SELECT ALL
        map.put(ActionType.SELECT_ALL, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                _table.selectAll();
            }
        }); // map.put
        
        // HIDE COLUMN
        map.put(ActionType.HIDE_COLUMN, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                int[] cols = _table.getSelectedColumns();
                convertColumnIndicesToModel (cols);
                for (int i = 0; i < cols.length; i++){
                    _parent.data.setHidden(cols[i], true);
                } // for
            } // actionPerformed
        }); // map.put
        
        // SHOW COLUMN
        map.put(ActionType.SHOW_COLUMN, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
               int[] cols = _table.getSelectedColumns();
               convertColumnIndicesToModel (cols);
               for (int i = 0; i < cols.length; i++){
                    _parent.data.setHidden(cols[i], false);
                } // for
            } // actionPerformed
        }); // map.put
        
        // CLEAR
        map.put(ActionType.CLEAR, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
               int[] cols = _table.getSelectedColumns();
               convertColumnIndicesToModel (cols);
               // We need to ensure that we delete a column only once,
               // and do it backwards, since deleting a column changes all
               // the indices above it
               boolean[] toDelete = new boolean[_parent.data.getColumnCount()];
               for (int i = 0; i < toDelete.length; i++) toDelete[i] = false;
               for (int i = 0; i < cols.length; i++) toDelete[cols[i]] = true;
               for (int i=0; i < toDelete.length; ++i){
                    if (toDelete[i]) _parent.data.removeColumn(i);
               } // for
             } // actionPerformed
        }); // map.put

    } // loadSpecificActions
    
    // loads data into the table from a newline/tab delimited string, in its
    // own Thread
    private void getDataFromString (final String text){
        if (text != null) {
            Runnable r = new Runnable(){
                public void run(){
                    DataGenerator data = new StringDataGenerator(text);
                    data.addListener (_parent);
                    data.start(null);
                } // run
            }; // new Runnable
            new Thread(r).start();
        } // if
    } // getDataFromString
    
    // converts an array of view indices (as from getSelectedColumns)
    // to an array of table indices
    private void convertColumnIndicesToModel (int[] cols){
        for (int i = 0; i < cols.length; i++){
            cols[i] = _table.convertColumnIndexToModel(cols[i]);
        } // for
    } // convertColumnIndicesToModel

} // TablePanel
