/* Tenua.java; the main entry point for the tenua program
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.KeyStroke;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentListener;
import javax.swing.text.Document;
import javax.swing.text.PlainDocument;
import tenua.parser.MechanismParser;
import tenua.simulator.DataGenerator;
import tenua.simulator.Mechanism;
import tenua.simulator.DataTableImpl;

/** A chemical kinetics simulator written in Java.
 *  Based on Barshop & Frieden's KINSIM.
 *  Each Tenua is a tabbed pane with:
 *  <ol>
 *  <li> a simple text editor that is compiled into a Mechanism that can be run
 *  <li> a list of the initial values of the variables in that mechanism
 *  <li> a graph of plotted data
 *  <li> a table of that same data,
 *  <li> a list of the last calculated values of the variables
 *  <li> a text editor for recording notes about the simulation
 *  </ol>
 *
 * <p>
 *  The mechanism itself is stored as a bound property of the JFrame's content pane
 *
 *  @author Daniel Wachsstock
 */
public class Tenua extends JTabbedPane 
  implements DocumentListener, DataGenerator.Listener, ChangeListener {

    // the number of Tenua windows so far created
    private static int _numWindows = 0;
    
    private EditorPanel _editor; // the editor panel; for setting the focus
    private final CompilerTimer _compiler = new CompilerTimer (this);
    // the thread running the current mechanism
    private Thread _mechanismThread = null;

    private int _numTabs = 0; // the number of tabs so far created
    private int _privateTabs; // the number of tabs created by the program, not
      // the user
    private boolean _showHidden = false; // flag whether to show hidden columns

    /** main entry point; creates one window */
    public static void main (String[] args){
        doNew();
    } // main
    
    public static void doNew(){
        javax.swing.SwingUtilities.invokeLater(new Runnable(){
            public void run(){
                final Tenua t = new Tenua();
                JFrame frame = new JFrame
                  ("Tenua--the kinetics simulator for Java");
                ++_numWindows;
                frame.getContentPane().setLayout (new BorderLayout());
                frame.getContentPane().add(t, BorderLayout.CENTER);
                frame.setLocation(_numWindows*20, _numWindows*20);
                // make sure we catch the window closing
                frame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
                frame.addWindowListener(new WindowAdapter(){
                    public void windowClosing(WindowEvent e) { t.dispose(); }
                }); // addWindowListener
                TenuaAction.createMenuBar(t, frame);
                TenuaAction.createToolBar(t, frame);
                frame.getContentPane().add 
                  (new JLabel (util.Resources.getString("About.line")),
                  BorderLayout.SOUTH);
                frame.setSize (700, 500);
                frame.setVisible(true);
                t._editor.acceptFocus();
            } // run
        }); // invokeLater
    } // doNew
    
    /** Create a new Tenua window */
    public Tenua(){
        super (javax.swing.SwingConstants.TOP);

        // create action map
        TenuaAction.loadActions(this);
        
        // create tabs
        _editor = new EditorPanel(this);
        addTab ("EditorTab", _editor);

        VariablePanel variablePanel = new VariablePanel(this, true);
        addTab ("InitialTab", variablePanel);
        addPropertyChangeListener(MECHANISM, variablePanel);

        addTab ("TableTab", new TablePanel(this));
        addTab ("PlotterTab", new PlotterPanel(this));
 
        variablePanel = new VariablePanel(this, false);
        addTab ("VariableTab", variablePanel);
        addPropertyChangeListener(MECHANISM, variablePanel);

        addTab ("NoteTab", new NotePanel(this));
        
        _privateTabs = _numTabs;
        

        // listen for changes to the document and the tabs
        mechanismText.addDocumentListener(this);
        addChangeListener (this);
        TenuaAction.setEnabledActions(this);
                
        setMechanism (null);
    } // constructor

    // set each tab and its shortcuts from a resource
    private void addTab (String resource, JPanel panel){
        String[] s = util.Resources.getString (resource).split(";");
        addTab (s[0], s[1], panel);
    } // addTab

    /** Add a new tab panel to the Tenua window
     *  @param name the name of the new tab
     *  @param tooltip the tool tip to display when the mouse rests on the tab
     *  @param panel the JPanel to put on this tab
     */
    public void addTab (String name, String tooltip, JPanel panel){
        final int n = _numTabs+1;
        String number = Integer.toString(n);
        super.addTab (name +" ("+number+")", panel);
        // add an action with key==number to select this tab
        getInputMap(WHEN_IN_FOCUSED_WINDOW).
          put(KeyStroke.getKeyStroke("alt " + number), number);
        getActionMap().put(number, new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                setSelectedIndex(n-1);
            } // actionPerformed
        }); // new AbstractAction
        setToolTipTextAt (_numTabs, tooltip);
        ++_numTabs;
    } // addTab
    
    /** remove tabs added by the user
     */
    public void removeUserTabs(){
        for (int i = getComponentCount(); i > _privateTabs; --i){
            removeTabAt(i-1);
            --_numTabs;
        }
    } // removeUserTabs
        
    /** run the mechanism */
    public void go(){
        try{
            if (getMechanism() == null) return;
            // if we are simulating, then don't restart
            if (_mechanismThread != null && _mechanismThread.isAlive()) return;
            _mechanismThread = new Thread(getMechanism());
            _mechanismThread.setDaemon(true); // allow quitting without waiting for the simulation to finish
            _mechanismThread.start();
        }catch (Exception ex){
            util.ErrorDialog.errorDialog("UnsupportedOperation", ex);
        }
    } // go
    
    /** stop the mechanism */
    public void stop(){
        if (_mechanismThread != null) _mechanismThread.interrupt();
    } // stop
    
    /** close the window */
    public void dispose(){
        boolean cancelled = false;
        /** ask each panel to close. We call the panel-specific action, and see if it set
         * its "closeCancelled" property to Boolean(true);
         */
        for (int i = 0; i < getComponentCount(); i++){
            Action closeAction = ((JPanel) getComponentAt(i))
              .getActionMap().get(ActionType.CLOSE);
            if (closeAction != null){
                closeAction.actionPerformed
                  (new ActionEvent(this,ActionEvent.ACTION_PERFORMED,"closeAttempt"));
                if (Boolean.TRUE.equals(closeAction.getValue("closeCancelled"))){
                    // cancelled; reset and mark
                    closeAction.putValue("closeCancelled", Boolean.FALSE);
                    cancelled = true;
                } // if closeAction.getValue
            } // if closeAction
        } // for
        if (cancelled) return;
        javax.swing.SwingUtilities.windowForComponent(this).dispose();
        _numWindows--;
        if (_numWindows == 0){
            System.exit(0);
        }
    } // dispose
    
    // the code for DataGenerator.Listener
    /** starts a mechanism by adding the output columns to the data table.
     */
    public void startingUp(DataGenerator source){
        java.util.Iterator i = source.getNames().iterator();
        for (i.next(); i.hasNext();){ // start at 2d output
            data.addColumn ((String) i.next());
        } // for
    } // startingUp

    /** adds new data to the data table
     *  @param v the data to add. v.get(0) should be the time
     *  and v.get(1..) should be the values
     */
    public void newData(DataGenerator source, nr.Vec v){
        // data vector has time as v[0] and outputs in v[1..]
        if (v.size() < 2) return; // need to plot time and an output
        double time = v.get(0);
        int index = 0;
        java.util.Iterator i = source.getNames().iterator();
        for (i.next(); i.hasNext();){
            String name = (String)i.next();
            double d = v.get(++index);
            if (Double.isNaN(d)) continue;
            int c = data.getColumnNumber (name);
            data.setY(time,c,d);
        } // for
    } // newData

    /** acknowleges the end of a simulation run
     *  by removing all empty rows
     */
    public void done(DataGenerator source){
        data.purgeRows();
    } // done

    // code for DocumentListener
    public void changedUpdate(javax.swing.event.DocumentEvent e) {
        /* do nothing; we only care about the text */
    }
    
    /** if the mechanism text changes, invalidates the compiled mechanism */
    public void insertUpdate(javax.swing.event.DocumentEvent e) {
        setMechanism(null);
    } // insertUpdate
    
    /** if the mechanism text changes, invalidates the compiled mechanism */
    public void removeUpdate(javax.swing.event.DocumentEvent e) {
        setMechanism(null);
    } // removeUpdate

    // code for ChangeListener
        /** handle activation/deactivation */
    public void stateChanged(javax.swing.event.ChangeEvent e) {
        TenuaAction.setEnabledActions (this);
    } // stateChanged
    
    /////////////// Getters and public final fields
    /////////////// final fields may as well be public if I am going to expose them
    /////////////// with a straight get method
 
    /** returns the active panel of the tabbed pane */
    public JPanel getActivePanel() {return (JPanel) getSelectedComponent();}
 
    /** returns the compiled mechanism.
     *  Is synchronized on <code>this</code> so the compiler thread and the Swing thread
     *  don't get out of sync.
     *  @return the current compiled mechanism
     */
    public synchronized Mechanism getMechanism() {
        return (Mechanism) getClientProperty(MECHANISM);
    }
 
    /** returns the number of Tenua windows open */
    public int getNumWindows () {return _numWindows;}
    
    /** Should hidden columns be displayed on the graph? */
    public boolean getShowHidden() {return _showHidden;}
    
    /** Set whether to display hidden columns on the graph */
    public void setShowHidden (boolean flag) {_showHidden = flag;}
    
    /** the data table for all the data */
    public final DataTableImpl data = new DataTableImpl();
    /** for displaying compiler errors */
    public final Document mechanismCompilerErrors = new PlainDocument();
    /** the text being edited to be compiled into a mechanism */
    public final Document mechanismText = new PlainDocument();
    /** the text of the notes panel */
    public final Document noteText = new PlainDocument();
    /** the key to use to get the mechanism from this Tenua window */
    public static final String MECHANISM = "mechanism";
    
    ///////////////
    // package-protected methods, basically used by the compiler
    // sets the compiler error messages, but in the Swing event thread
    void setCompilerErrors (final String s){
        javax.swing.SwingUtilities.invokeLater ( new Runnable(){
            public void run(){
                try{
                    mechanismCompilerErrors.remove
                      (0, mechanismCompilerErrors.getLength());
                    mechanismCompilerErrors.insertString
                      (0, s, null);  
                }catch (Exception ex) { /* do nothing */ }
            } // run
        }); // invokeLater
    } // setCompilerErrors        
 
    /** Sets the compiled mechanism.
     *  Is synchronized on <code>this</code> so the compiler thread and the Swing thread
     *  don't get out of sync.
     *  @param mechanism the new {@link Mechanism}.
     */
    public synchronized void setMechanism (final Mechanism mechanism){
        if (mechanism == null){
            _compiler.compile();
        }else{
            mechanism.addListener(this);
        }
        javax.swing.SwingUtilities.invokeLater(new Runnable(){
            public void run(){
                if (mechanism != null){
                    mechanism.put ("dataTable", data);
                    mechanism.put ("notes", noteText);
                    mechanism.put ("tenua", Tenua.this);
                } // if                
                Tenua.this.putClientProperty(MECHANISM, mechanism);
                TenuaAction.enable
                   (Tenua.this, ActionType.GO.name(), mechanism != null);
            } // run
        }); // invokeLater
    } // setMechanism

} // Tenua
    