/*
 * TenuaAction.java
 *
 *
 * Created on May 18, 2004, 1:10 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.print.PageFormat;
import java.awt.print.PrinterJob;
import javax.swing.*;

/** TenuaAction.java
 *  Action subclass that remembers which Tenua window its associated with
 *  static methods to load and action map, menubar and toolbar from 
 *  a resource file for a Tenua window
 *
 *  adds keys for "codeName" (the internal, not localized, name for the action)
 *  and "parent" (the Tenua window that created it).
 *  the PageSetup action also has a key "pageFormat" for the PageFormat object
 * @author  Daniel Wachsstock
 */
abstract class TenuaAction extends AbstractAction {
    
    // all the actions that are to be defined by each individual panel
    static final ActionType[] _specificActions = {ActionType.LOAD, ActionType.SAVE,
      ActionType.SAVE_AS, ActionType.PRINT, ActionType.UNDO, ActionType.REDO,
      ActionType.CUT, ActionType.COPY, ActionType.PASTE, ActionType.CLEAR,
      ActionType.SELECT_ALL, ActionType.HIDE_COLUMN, ActionType.SHOW_COLUMN };
      
    /** create all the actions needed for this program from the resource bundle
     *  and puts them into the content pane's (which is a JComponent, but not
     *  declared that way) ActionMap, with up to 2 keys each--the localized name and
     *  the code name.
     */
    public static void loadActions (final Tenua parent){
        TenuaAction a;
        ActionMap map = parent.getActionMap();

        // add the specific actions
        for (int i = 0; i < _specificActions.length; i++){
            a = new SpecificAction (_specificActions[i]);
            setActionStrings (a, _specificActions[i].name(), parent);       
            addActionToMap (a, map);            
        }
        
        // Add the unversal actions
        // New
        a = new TenuaAction(){
            public void actionPerformed(ActionEvent e){
                parent.doNew();
            }
        };
        setActionStrings (a, ActionType.NEW.name(), parent);
        addActionToMap (a, map);
        // Page Setup
        a = new TenuaAction(){
            public void actionPerformed (ActionEvent e){
                PageFormat fmt = (PageFormat) getValue ("pageFormat");
                if (fmt == null) fmt = PrinterJob.getPrinterJob().defaultPage();
                fmt = PrinterJob.getPrinterJob().pageDialog(fmt);
                putValue ("pageFormat", fmt);
            }
        };
        setActionStrings (a, ActionType.PAGE_SETUP.name(), parent);
        addActionToMap (a, map);
        // Close
        a = new TenuaAction(){
            public void actionPerformed(ActionEvent e){
                // note: parent.dispose looks for JPanel.getActionMap().get(ActionType.CLOSE)
                // for each tabbed panel, so it can do cleanup.
                // if the panel sets a key of "closeCancelled" to Boolean.TRUE
                // then the whole dispose is cancelled.
                parent.dispose();
            }
        };
        setActionStrings (a, ActionType.CLOSE.name(), parent);
        addActionToMap (a, map);
        // Exit
        a = new TenuaAction(){
            public void actionPerformed(ActionEvent e){
                // TODO: close each window
                System.exit(0);
            }
        };
        setActionStrings (a, ActionType.EXIT.name(), parent);
        addActionToMap (a, map);
        // Add Row
        a = new TenuaAction(){
            public void actionPerformed (ActionEvent e){
                String title = util.Resources.getString ("AddRowDialogTitle");
                String prompt = util.Resources.getString ("AddRowDialogPrompt");
                String str = JOptionPane.showInputDialog
                  (parent, prompt, title, JOptionPane.PLAIN_MESSAGE);
                if (str != null){
                    String[] rows = str.split(";");
                    for (int i=0;i<rows.length;++i) try{
                        parent.data.setY
                          (Double.parseDouble(rows[i]), -1, 0);
                    }catch (NumberFormatException ex){
                        /* ignore */
                    } // for try
                } // if
            } // new TenuaAction
        };
        setActionStrings (a, ActionType.ADD_ROW.name(), parent);
        addActionToMap (a, map);
        // Add Column
        a = new TenuaAction(){
            public void actionPerformed (ActionEvent e){
                String title = util.Resources.getString ("AddColumnDialogTitle");
                String prompt = util.Resources.getString ("AddColumnDialogPrompt");
                String str = JOptionPane.showInputDialog
                  (parent, prompt, title, JOptionPane.PLAIN_MESSAGE);
                if (str != null){
                    String[] cols = str.split(";");
                    for (int i=0;i<cols.length;++i){
                        parent.data.addColumn(cols[i]);
                    } // for
                }
            }
        };
        setActionStrings (a, ActionType.ADD_COLUMN.name(), parent);
        addActionToMap (a, map);
        // Show Hidden
        a = new TenuaAction(){
            public void actionPerformed (ActionEvent e){
                boolean showHidden = !parent.getShowHidden();
                parent.setShowHidden (showHidden);
                if (showHidden){
                    putValue (Action.NAME, util.Resources.getString("HideHiddenName"));
                }else{
                    putValue (Action.NAME, util.Resources.getString("ShowHiddenName"));
                }
            } // actionPerformed
        };
        setActionStrings (a, ActionType.SHOW_HIDDEN.name(), parent);
        addActionToMap (a, map);
        // Go
        a = new TenuaAction(){
            public void actionPerformed (ActionEvent e){
                parent.go();
            }
        };
        setActionStrings (a, ActionType.GO.name(), parent);
        addActionToMap (a, map);
        // Stop
        a = new TenuaAction(){
            public void actionPerformed (ActionEvent e){
                parent.stop();
            }
        };
        setActionStrings (a, ActionType.STOP.name(), parent);
        addActionToMap (a, map);
        // ShowManual
        a = new TenuaAction(){
            public void actionPerformed (ActionEvent e){
                HelpWindow.showHelp();
            }
        };
        setActionStrings (a, ActionType.SHOW_MANUAL.name(), parent);
        addActionToMap (a, map);
        // ShowManual
        a = new TenuaAction(){
            public void actionPerformed (ActionEvent e){
                AboutWindow.about();
            }
        };
        setActionStrings (a, ActionType.ABOUT.name(), parent);
        addActionToMap (a, map);
    } // loadActions

    /** return the PageFormat for this Tenua window
     *  @param parent the Tenua window whose page setup format to get
     *  @return the page setup... PageFormat
     */
    public static PageFormat getFormat (Tenua parent){
        ActionMap map = parent.getActionMap();
        Action pageSetup = map.get(ActionType.PAGE_SETUP.name());
        PageFormat format = null;
        if (pageSetup != null){
            format = (PageFormat) pageSetup.getValue ("pageFormat");
        } // if
        return format;
    } // getFormat
    
    /** create the menubar from the resource file.
     *  the menu bar has key=MenuBar, with ';' separated codeNames of the menus
     *  each menu has the codeName as the key, with format name;accelerator key;actions,
     *  where actions is a ';' separated list of action codeNames. a blank represents a
     *  separator
     *  @param parent the Tenua window to own this menu bar
     */
    static void createMenuBar (final Tenua parent, final JFrame window){
        ActionMap map = parent.getActionMap();
        String[] menus = util.Resources.getString("MenuBar").split(";");
        JMenuBar menuBar = new JMenuBar();
        for (int i = 0; i < menus.length; i++){
            if (menus[i].length() == 0) continue;
            String[] menuDetails = util.Resources.getString(menus[i]).split(";");
            if (menuDetails.length == 0) continue;
            JMenu menu = new JMenu(menuDetails[0]);
            if (menuDetails.length > 1){
                menu.setMnemonic(getKeyCode(menuDetails[1]));
            } // if
            for (int j = 2; j < menuDetails.length; j++){
                // go through the list of actions
                if (menuDetails[j].length() > 0){
                    Action a = map.get(menuDetails[j]);
                    if (a != null) menu.add(new JMenuItem(a));
                }else{
                    menu.addSeparator();
                } // if
            } // for
            menuBar.add(menu);
        } // for
        window.setJMenuBar(menuBar);
    } // createMenuBar

    /** create the toolbar from the resource file.
     *  the menu bar has key=ToolBar, with ';' separated codeNames of the actions
     *  A blank represents a separator
     *  @param parent the Tenua window to own this tool bar
     */
    static void createToolBar (final Tenua parent, final JFrame window){
        ActionMap map = parent.getActionMap();
        String[] actions = util.Resources.getString("ToolBar").split(";");
        JToolBar toolBar = new JToolBar();
        InputMap input = toolBar.getInputMap();
        for (int i = 0; i < actions.length; i++){
            // go through the list of actions
            if (actions[i].length() > 0){
                Action a = map.get(actions[i]);
                if (a != null) toolBar.add(a);
            }else{
                toolBar.addSeparator();
            } // if
        } // for
        window.getContentPane().add(toolBar, java.awt.BorderLayout.PAGE_START);
    } // createToolBar
    
    // used for actions that must be defined by each individual panel
    static class SpecificAction extends TenuaAction {
        private final ActionType _type;
 
        public SpecificAction (ActionType type) {_type = type;}

        public void actionPerformed(ActionEvent e){
            Tenua parent = (Tenua) getValue ("parent");
            JComponent panel = parent.getActivePanel();
            ActionMap map = panel.getActionMap();
            Action a = map.get (_type);
            if (a != null){
                a.actionPerformed(e);
            } // if
        } // actionPerformed       
    } // SpecificAction
    
    // set the action key/values from the resource file
    // format is name;mnemonic;accelerator;icon;short description
    // also adds the parent Tenua window with key "parent"
    // and the internal name under "codeName"
    static void setActionStrings(Action a, String name, Tenua parent){
        final int MENU_MASK = parent.getToolkit().getMenuShortcutKeyMask(); 
        try{
            a.putValue ("parent", parent);
            a.putValue ("codeName", name);
            String[] strings = util.Resources.getString(name).split(";");
            if (strings.length > 0 && strings[0] != null){
                a.putValue(a.NAME, strings[0]);
            }
            if (strings.length > 1 && strings[1] != null && !strings[1].equals("")){
                a.putValue(a.ACCELERATOR_KEY, KeyStroke.getKeyStroke(getKeyCode(strings[1]), MENU_MASK));
            }
            if (strings.length > 2 && strings[2] != null && !strings[2].equals("")){
                a.putValue(a.MNEMONIC_KEY, new Integer(getKeyCode(strings[2])));
            }
            if (strings.length > 3 && strings[3] != null && !strings[3].equals("")){
                String location = util.Resources.resourceDir + strings[3];
                // get resources from the top of the classpath
                java.net.URL url = ClassLoader.getSystemResource(location);
                a.putValue(a.SMALL_ICON, new ImageIcon(url));
            }
            if (strings.length > 4 && strings[4] != null){
                a.putValue(a.SHORT_DESCRIPTION, strings[4]);
            }
        }catch (Exception ex) {
            Object[] o = {name};
            util.ErrorDialog.errorDialogSpecific("ActionCreationFailure", o, ex);
        } // catch
    } // setActionStrings
    
    // ugly hack to get KeyEvent keycodes from a resource file. There ought to be a better way.
    // This creates the VK_ field and calls it.
    static int getKeyCode(String keyName){
        try{
            int result = KeyEvent.class.getDeclaredField("VK_"+keyName).getInt(null);
            return result;
        }catch (Exception ex){
            return KeyEvent.VK_UNDEFINED;
        } // try
    } // getKeyCode

    // adds the action to the action map with both keys
    static void addActionToMap (Action a, ActionMap map){
        Object name = a.getValue (a.NAME);
        if (name != null) map.put (name, a);
        name = a.getValue ("codeName");
        if (name != null) map.put (name, a);
    }
    
    // enables or disables a specific Action
    static void enable (Tenua parent, Object key, boolean isEnabled){
        ActionMap map = parent.getActionMap();
        Action a = map.get (key);
        if (a != null) a.setEnabled (isEnabled);
    } // enable

    // enables or disables all the specific actions for a tab
    static void setEnabledActions (Tenua parent){
        ActionMap map = parent.getActivePanel().getActionMap();
        for (int i = 0; i < _specificActions.length; i++){
            // enable it if the specific action is not null
            enable (parent, _specificActions[i].name(),
              map.get(_specificActions[i]) != null);
        } // for
    } // setEnabledActions
    
} // TenuaAction

/** just a marker for different actions
 * using these instead of strings as keys
 * means it will never conflict with preexisting actions
 */
class ActionType {
    private final String _name;
    public ActionType (String name) {_name = name;}
    public String name() {return _name;}
    
    static final ActionType NEW = new ActionType("New");
    static final ActionType LOAD = new ActionType("Load");
    static final ActionType SAVE = new ActionType("Save");
    static final ActionType SAVE_AS = new ActionType("SaveAs");
    static final ActionType PRINT = new ActionType("Print");
    static final ActionType PAGE_SETUP = new ActionType("PageSetup");
    static final ActionType CLOSE = new ActionType("Close");
    static final ActionType EXIT = new ActionType("Exit");
    static final ActionType UNDO = new ActionType("Undo");
    static final ActionType REDO = new ActionType("Redo");
    static final ActionType CUT = new ActionType("Cut");
    static final ActionType COPY = new ActionType("Copy");
    static final ActionType PASTE = new ActionType("Paste");
    static final ActionType CLEAR = new ActionType("Clear");
    static final ActionType SELECT_ALL = new ActionType("SelectAll");
    static final ActionType ADD_ROW = new ActionType("AddRow");
    static final ActionType ADD_COLUMN = new ActionType("AddColumn");
    static final ActionType HIDE_COLUMN = new ActionType("HideColumn");
    static final ActionType SHOW_COLUMN = new ActionType("ShowColumn");
    static final ActionType SHOW_HIDDEN = new ActionType("ShowHidden");    
    static final ActionType COMPILE = new ActionType("Compile");
    static final ActionType GO = new ActionType("Go");
    static final ActionType STOP = new ActionType("Stop");
    static final ActionType SHOW_MANUAL = new ActionType("ShowManual");
    static final ActionType ABOUT = new ActionType("About");
} // ActionType

