/*
 * VariablePanel.java
 *
 * Created on July 9, 2004, 4:38 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.gui;
import tenua.parser.MechanismParser;
import tenua.parser.ParseException;
import tenua.simulator.DataGenerator;
import tenua.simulator.Mechanism;
import tenua.symbol.SymbolTable;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.DataFlavor;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import javax.swing.*;
import javax.swing.border.EtchedBorder;

/** Implements a panel that displays the variables of a mechanism
 *
 * @author  Daniel Wachsstock
 */
public class VariablePanel extends JPanel
  implements DataGenerator.Listener, PropertyChangeListener {
    private final Tenua _parent;
    private final boolean _isInitial;
    
    /** Creates a new instance of VariablePanel */
    public VariablePanel (Tenua parent, boolean isInitial) {
        super();
        _parent = parent;
        _isInitial = isInitial;
        setLayout (new BorderLayout());
        loadSpecificActions();
    }
        
    public void done(DataGenerator source) {}
    
    public void newData(DataGenerator source, nr.Vec data) { repaint(); }
    
    public void startingUp(DataGenerator source) { repaint(); }
    
    public void propertyChange(PropertyChangeEvent evt) {
        if (Tenua.MECHANISM.equals(evt.getPropertyName())){
            Mechanism mechanism = _parent.getMechanism();
            if (mechanism != null) mechanism.addListener(this);
            removeAll(); // clean this panel out
            Box box = Box.createHorizontalBox();
            box.setBorder(BorderFactory.createLineBorder(Color.darkGray));
            box.add (new VariablePanel.MechanismList
              (mechanism, Mechanism.TIME_CONSTANTS));
            box.add (new VariablePanel.MechanismList
              (mechanism, Mechanism.SPECIES));
            box.add (new VariablePanel.MechanismList
              (mechanism, Mechanism.RATE_CONSTANTS));
            box.add (new VariablePanel.MechanismList
              (mechanism, Mechanism.PARAMETERS));
            if (_isInitial && mechanism != null) box.add (new ResetButton());
            add (new JScrollPane (box));
            validate(); // tell the panel to layout all the stuff
        }
    } // propertyChange
    
    /** puts the data in a String that can be re-compiled to get the data back.
     *  @return the String created
     */
    public String toString(){
        StringBuffer result = new StringBuffer();
        Mechanism mechanism = _parent.getMechanism();
        if (mechanism == null) return result.toString();
        appendList (result, mechanism, Mechanism.TIME_CONSTANTS);
        appendList (result, mechanism, Mechanism.SPECIES);
        appendList (result, mechanism, Mechanism.RATE_CONSTANTS);
        appendList (result, mechanism, Mechanism.PARAMETERS);
        return result.toString();
    } // toString
    
    // adds the variables in the variableNames(listName) list
    // to s
    protected void appendList (StringBuffer s, Mechanism mechanism, String listName){
        List list = mechanism.variableNames(listName);
        s.append("// ").append(listName).append("\n");
        for (Iterator i = list.iterator(); i.hasNext();){
            String name = (String) i.next();
            s.append(name).append(" = ");
            s.append (_isInitial ?
              mechanism.getInitial(name) :
              mechanism.getLatest(name));
            s.append(";\n");
        } // for
    } // appendList
    
    /** runs a macro that contains variable-setting commands (like a=1;).
     *  @param s the macro to run
     *  @throws tenua.parser.ParseException if the macro cannot be compiled
     *  @throws bsh.EvalError if the macro cannot be evaluated
     */
    public void getDataFromString (String s)
      throws ParseException, bsh.EvalError {
        if (s == null) return;
        Mechanism mechanism = _parent.getMechanism();
        if (mechanism == null) return;
        // have the compiler convert mechanism variables to beanshell
        // put/get calls
        s = CompilerTimer.compileMacro(s, fakeSymbolTable (mechanism));
        // execute the resulting string
        mechanism.eval (s);
        repaint();
    } // getDataFromString

    // creates a temporary SymbolTable to allow variable-name
    // mangling by the compiler. The alternative would have been to
    // implement a Mechanism.getSymbolTable() method, which would violate
    // encapsulation something fierce.
    protected SymbolTable fakeSymbolTable (Mechanism mechanism){
        SymbolTable result = new SymbolTable();
        addSymbols (result, mechanism, Mechanism.TIME_CONSTANTS);
        addSymbols (result, mechanism, Mechanism.SPECIES);
        addSymbols (result, mechanism, Mechanism.RATE_CONSTANTS);
        addSymbols (result, mechanism, Mechanism.PARAMETERS);
        return result;
    } // fakeSymbolTable

    protected void addSymbols (SymbolTable st, Mechanism mechanism, String listName){
        List list = mechanism.variableNames(listName);
        for (Iterator i = list.iterator(); i.hasNext();){
            st.put((String) i.next());
        } // for        
    } // addSymbols
    
    // adds appropriate actions to this action map
    private void loadSpecificActions(){
        final javax.swing.ActionMap map = this.getActionMap();

        // LOAD
        map.put(ActionType.LOAD, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                try{
                    getDataFromString(util.StringFiler.load(null));
                }catch (Exception ex){
                    util.ErrorDialog.errorDialog("Variable.Load", ex);
                }
            } // actionPerformed
        }); // map.put

        // SAVE
        // saves the data
        map.put(ActionType.SAVE, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                String text = VariablePanel.this.toString();
                util.StringFiler.save(text, null);
            } // actionPerformed
        }); // map.put

        // SAVE_AS
        // same as SAVE for this (no default file name)
        map.put(ActionType.SAVE_AS, map.get (ActionType.SAVE));
        
        // PRINT
        map.put (ActionType.PRINT, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                util.StringPrinter job = new util.StringPrinter
                  (VariablePanel.this.toString(), TenuaAction.getFormat(_parent));
                String title = ((JFrame)SwingUtilities.windowForComponent
                  (_parent)).getTitle();
                job.setTitle (title + " " +
                   util.Resources.getString(_isInitial ?
                   "InitialTab" : "VariableTab").split(";")[0]);
                job.start();
            } // actionPerformed
        }); // map.put

        // COPY
        map.put(ActionType.COPY, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                StringSelection text = new StringSelection
                  (VariablePanel.this.toString());
                VariablePanel.this.getToolkit().getSystemClipboard().
                  setContents(text, text);
            } // actionPerformed
        }); // map.put

        // PASTE
        map.put(ActionType.PASTE, new javax.swing.AbstractAction(){
            public void actionPerformed (java.awt.event.ActionEvent e){
                java.awt.datatransfer.Transferable t = 
                  VariablePanel.this.getToolkit().
                  getSystemClipboard().getContents(this);
                try{
                    VariablePanel.this.getDataFromString ((String)
                      t.getTransferData(DataFlavor.stringFlavor));
                }catch (Exception ex) {
                  util.ErrorDialog.errorDialog("Variable.Paste", ex);
                }
            } // actionPerformed
        }); // map.put

    } // loadSpecificActions

    // displays a list of Mechanism variables
    // like with TablePanel, it seems that I cannot control the size of JComponents
    // consistently; I have to put everything in a panel and use that.
    class MechanismList extends JPanel{
        private final Dimension _size;
        
        public MechanismList (Mechanism mechanism, String listName){
            super();
            Box box = Box.createVerticalBox();
            setBorder(BorderFactory.createTitledBorder
              (BorderFactory.createEtchedBorder(EtchedBorder.LOWERED), listName));
            // make sure we have room for the title
            JLabel testLabel = new JLabel(listName);
            box.add (Box.createHorizontalStrut(testLabel.getPreferredSize().width));
            add (box);
            _size = testLabel.getPreferredSize();
            if (mechanism == null) return;
            List list = mechanism.variableNames(listName);
            for (Iterator i = list.iterator(); i.hasNext();){
                String name = (String) i.next();
                box.add(new MechanismTextField(mechanism, name, _isInitial));
            } // for
        } // constructor
        
        public Dimension getMinimumSize() {return _size; }
    } // MechanismList
    
    class ResetButton extends JPanel{
        public ResetButton(){
          JButton button = new
            JButton(util.Resources.getString("ResetButton.label"));
          button.addActionListener(new java.awt.event.ActionListener(){
                public void actionPerformed (java.awt.event.ActionEvent e){
                    Mechanism m = _parent.getMechanism();
                    if (m != null){
                        m.resetVariables();
                        VariablePanel.this.repaint();
                    }
                } // actionPerformed
            });
            button.setToolTipText(util.Resources.getString("ResetButton.tooltip"));
            add(button);
        }
        
        public Dimension getMaximumSize() {return getPreferredSize();}
    } // ResetButton
}
