/*
 * DataGenerator.java
 *
 * Created on May 13, 2004, 7:40 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.simulator;

/** Interface that represents something that generates data,
 *  in the form of an array of doubles; this is a copy of the internal
 *  data and can be modified at will. The same number of output data are generated at each pass
 *
 *  This is a natural to put in an independent Thread, so
 *  an implementation should (but I can't find anyway to enforce this) call Thread.yield
 *  at least once through every pass, and check for Thread.isInterrupted and stop generating
 *  data, and call done on each listener if true.
 *
 *  An implementation may support getting a Memento of its internal state
 *  (See Design Patterns) to restart the data generator or watch it working
 *  if it also exposes the format of the Memento
 *
 * @author  Daniel Wachsstock
 */
public interface DataGenerator extends Runnable {

    /** starts (or restarts) the data generator
     *  @param memento a Memento of the internal state of the data generator, or null
     *  to start afresh. Calls startingUp on each listener either way
     *  <p>
     *  a Memento returned by {@link #getMemento()} for the same DataGenerator is guarranteed
     *  not to throw and will continue the data generation from "close" to where it was
     *  generated. "close" is implementation-specific
     *  @throws UnsupportedOperationException if a non-null Memento is passed and
     *  this implementation does not support Mementos
     *  @throws IllegalArgumentException if the Memento is not supported by this DataGenerator
     */
    public void start(Object memento);
    
    /** @return the names of the output data. If there are no names, it
     *  will be an list of empty strings (not null!),
     *  which will at least give the caller the number of outputs
     */
    public java.util.List getNames();

    /** @return the current internal state; what this means is
     *  implementation specific. It also does not guarrantee that
     *  the data generator can be started up in exactly the same state it
     *  is in now, just that it will be "close" in some sense and
     *  it would be possible to reset it to that Memento with {@link #start(Object)}
     *  @throws UnsupportedOperationException if this implementation does not support Mementos
     */
    public Object getMemento();
    
    /** add a DataGenerator.Listener to this DataGenerator's list */
    public void addListener (Listener listener);

    /** remove a given DataGenerator.Listener from this DataGenerator's list */
    public void removeListener (Listener listener);
    
    /** a class that listens for data being generated */
    public interface Listener{
            /** called before data is generated */
            void startingUp(DataGenerator source);
            /** called with each new dataset, with a <b>copy</b> of the data
             *  so the listener can modify it at will. */
            void newData (DataGenerator source, nr.Vec data);
            /** called after data is finished */
            void done(DataGenerator source);
    } // Listener
    
} // DataGenerator