/*
 * DataTable.java
 *
 * Created on September 11, 2005, 10:08 AM
 *
 *  Copyright 2005 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.simulator;

/** A two-dimensional table with x values for the rows and column names for
 *  the columns, and y values as the data in the columns (i.e. a set of data
 *  series with identical x-values). Columns are allowed to have missing data;
 *  the method {@link getY} extrapolates from the values at higher and lower
 *  x's. (@link getLowestx}, {@link getHighestX}, and {@link getNextX} only
 *  return x values for which the given column has data; they can be used to
 *  implement an iterator over actual data in a column.
 *  The interface is designed to be made thread-safe with the Swing event
 *  thread; reader methods should wait for writers
 *  and writer threads should
 *  run the actual data-modifying routines on the Swing event thread.
 */

public interface DataTable {

    /** Find the column index for a given column.
     *  @param name the column name to look for
     *  @returns the index of the column
     *  @throws IllegalArgumentException if the column does not exist
     */
    public int getColumnNumber (String name);

    /** return the y value of a given x value and column. Interpolates
     *  linearly if there is no data with that x value in that column.
     *  @param x the X-value
     *  @param c the column number (column -1 is the X-value)
     *  @return the y value for that row
     *  @throws ArrayIndexOutOfBoundsException if the row or column do not exist
     **/
    public double getY(double x, int c);

    /** returns the lowest x value for which a column has data
     *  @param c the column to search
     *  @return the lowest x value. If there is no data in the column,
     *  returns Double.POSITIVE_INFINITY
     *  @throws ArrayIndexOutOfBoundsException if the column does not exist
     **/   
    public double getLowestX (int c);

    /** returns the highest x value for which a column has data
     *  @param c the column to search
     *  @return the highest x value. If there is no data in the column,
     *  returns Double.NEGATIVE_INFINITY
     *  @throws ArrayIndexOutOfBoundsException if the column does not exist
     **/
    public double getHighestX (int c);

    /** returns the next x value for which a column has data
     *  @param c the column to search
     *  @param x the previous x value
     *  @return the next x value. If there is no more data in the column,
     *  returns Double.POSITIVE_INFINITY
     *  @throws ArrayIndexOutOfBoundsException if the column does not exist
     **/   
    public double getNextX (int c, double x);

    /** returns the nth x value for which data exists in a column. Indexed
     *  from zero.
     *  @param n which x value to return
     *  @param c the column to search
     *  @return the x value
     *  @throws ArrayIndexOutOfBoundsException if the column does not exist
     *  @throws IllegalArgumentException if there are less than n data points
     */
    public double getNthX (int c, int n);

    /** returns the number of data points in a column
     *  @param c the column to search
     *  @return the number of points with data
     *  @throws ArrayIndexOutOfBoundsException if the column does not exist
     */
    public double getCount (int c);

    /** Set the y value of a given x-value and column. If the x-value
     *  does not exist, it is inserted at the appropriate place in the table.
     *  Specifying column -1 inserts a blank row in the table (the y value
     *  is ignored
     *  @param x the x value
     *  @param c the column number
     *  @param y the new value
     *  @throws ArrayIndexOutOfBoundsException if the row or column do not exist
     **/
    public void setY (final double x, final int c, final double y);
    
    /** Adds a column or erases it if it already exists
     *  @param name the name of the new column
     */
    public void addColumn (final String name);

} // DataTable