/*
 * DataTableImpl.java
 *
 * Created on September 5, 2005, 10:08 AM
 *
 *  Copyright 2005 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */


package tenua.simulator;
import java.lang.reflect.InvocationTargetException;
import javax.swing.SwingUtilities;
import javax.swing.table.*;

/** A two-dimensional TableModel. Contains only double data, with an extra
 *  column of "X values" that can be put into a row header panel in a 
 *  JScrollPane as:
 *  <pre>
 *      DataTable model = new DataTable();
 *      JTable mainTable = new JTable(model);
 *      JScrollPane pane = new JScrollPane(mainTable);
 *      JTable rowHeaders = new JTable(model.getRowHeaderModel());
 *      pane.setRowHeaderView(new JPanel(new BorderLayout()).add (rowHeaders));
 *  </pre>
 *  The trick with putting the rowHeaders into its own panel seems to be the
 *  only way I can get it to size correctly.
 *  <br>
 *  It is intended to be used as:
 *  <pre>
 *      model.addColumn("first");
 *      model.addColumn("second");
 *      col1 = model.getColumnNumber("first");
 *      double x = 10; y = 50;
 *      model.setY (x, col1, y);
 *  </pre>
 *  where <code>x</code> is a double representing the x value. You can use
 *  the TableModel methods of <code>setValueAt (row, col, new Double (y))</code>
 *  if you know the exact row you want to set, or the thread-safe versions
 *  like <code>setY (row, col, y)</code>.
 *  <br>
 *  It also supports setting a "hidden" flag that makes the cell uneditable.
 *  Column names are unique. Adding a column whose name already exists erases
 *  that column.
 *  <H3>Thread safety</H3>
 *  This class is the interface between the GUI, which runs on the Swing event
 *  thread, and the calculating part of the program, which may run on many
 *  threads. To ensure that all reads and writes are current, the writer methods
 *  only execute on the Swing event thread (they use SwingUtilities.invokeLater).
 *  Readers wait until all pending writers are done.
 *  <br>
 *  The thread-safe writers are:
 *  <ul>
 *  <li>{@link setYAt}
 *  </ul>
 *  The thread-safe readers are:
 *  <ul>
 *  <li>{@link getXAt}
 *  <li>{@link getYAt}
 *  <ul>
 *  The TableModel and AbstractTableModel methods
 *  (findColumn, getColumnClass, getColumnCount, getColumnName, getRowCount,
 *  getValueAt, isEditable, setValueAt) are all <b>not</b> thread-safe; they
 *  must be run on the Swing event thread.
 *  <H3>Notes</H3>
 *  <ul>
 *  <li>Rows remain sorted by X-value. Columns remain in the order they were
 *  created.
 *  <li>Methods that use the word "At" index rows by row number. Other methods
 *  use the X-value to determine the row.
 *  <li>"Get" methods that use the row number return Double.NaN if there is no
 *  data in that particular cell. Methods that use the X-value interpolate if
 *  there is no data in that cell, from the cell with X-value just less than
 *  the desired x and the cell just greater; it is a linear interpolation.
 *  If the desired x is less than all the available x's, the value returned is
 *  that of the cell with the lowest x. Similarly, if the desired x is too high,
 *  the value of the cell with the highest x is returned. If there is no data
 *  in the column, 0 is returned.
 *  <li>For all methods, column number -1 refers to the X-values. So
 *  <code>setY (x, -1, any-value)</code> inserts a row with X-value x and no
 *  data in any of the columns.
 *  </ul>
 *
 * @author Daniel Wachsstock
 */
public class DataTableImpl extends AbstractTableModel implements DataTable {
    
      // the first column (_data[][0]) is the X-values; the rest
      // are the Y-values
    protected double[][] _data;
    protected String[] _names;
    protected boolean[] _hidden; // true if the column is hidden
      // the next two are the number of rows and columns actually used
      // _columnCount is the number of data columns, not including the
      // X-value column
    protected int _rowCount;
    protected int _columnCount;
      // the next two are the size of the arrays
    protected int _rowSize;
    protected int _columnSize;
    protected final AbstractTableModel _rowHeaders;
      // The row that had the x-value most recently looked for with findRow.
      // The next time we look, we will likely search close to that.
    private int _lastRowSought = 0;
    
    protected final int INITIAL_ROW_SIZE=10;
    protected final int INITIAL_COLUMN_SIZE=10;
    
    
    /** Creates a new instance of DataTable */
    public DataTableImpl() {
        _data = new double[INITIAL_ROW_SIZE][INITIAL_COLUMN_SIZE];
        _rowSize = _data.length;
        _columnSize = _data[0].length;
        _rowCount = 0;
        _columnCount = 0;
        _rowHeaders = new DataTableHeaders();
        _names = new String [INITIAL_COLUMN_SIZE];
        _names[0]="X";
        _hidden = new boolean [INITIAL_COLUMN_SIZE];
        _hidden[0] = true;
    } // constructor
    
    //*******
    // Methods for TableModel. Not threadsafe if used outside
    // the swing event Thread
    public int findColumn (String name){
        for (int c=1; c<=_columnCount; ++c)
          if (_names[c].equals(name)) return c-1;
        return -1;
    } // findColumn
    public Class getColumnClass(int c) {return Double.class;}
    public int getColumnCount() {return _columnCount;}
    public String getColumnName(int c) {return _names[c+1];}
    public int getRowCount() {return _rowCount;}
    public TableModel getRowHeaderModel() {return _rowHeaders;}
    public Object getValueAt (int r, int c) {
        return new Double (_data[r][c+1]);
    } // getValueAt
    public boolean isCellEditable(int r, int c){
        if (c == -1 || _hidden[c+1]) return false;
        return true;
    }// isCellEditable
    public void setColumnName (int c, String name) {
        _names[c+1]=name;
        fireTableStructureChanged();
    }
    public void setValueAt (Object o, int r, int c){
        if (o instanceof Number){
            _data[r][c+1] = ((Number)o).doubleValue();
        }else try {
            _data[r][c+1] = Double.parseDouble(o.toString());
        }catch (Exception ex){
            _data[r][c+1] = Double.NaN;
        }
        fireTableCellUpdated (r, c);
    } // setValueAt
    
    /** a string representation of this table
     *  @return a tab and newline delimited table
     */
    public String toString(){
        // select all the columns to send to toString(int[])
        int[] cols = new int[_columnCount];
        for (int i = 0; i < cols.length; i++) cols[i] = i;
        return toString (cols);
    } // toString
    
    //*******
    // Reader methods
    // All methods look like:
    //      waitForWriters();
    //      return something;
    
    /** return the x value of a given row
     *  @param r the row number
     *  @return the x value for that row
     *  @throws ArrayIndexOutOfBoundsException if the row does not exist
     **/
    public double getXAt(int r){
        waitForWriters();
        return _data[r][0];
    } // getXAt
    
    /** return the y value of a given row and column
     *  @param r the row number
     *  @param c the column number (column -1 is the X-value)
     *  @return the y value for that row
     *  @throws InterruptedException if the routine is interrupted
     *  before it gets a chance to read the data
     *  @throws ArrayIndexOutOfBoundsException if the row or column do not exist
     **/
    public double getYAt(int r, int c){
        waitForWriters();
        return _data[r][c+1];
    } // getYAt
        
    public double getY(double x, int c){
        int lo, hi;
        double result;
        waitForWriters();
        int r = findRow (x);
        result = _data[r][c+1];
        if (Double.isNaN(result)) {
            // not a real number
            // search for a number less than x
            for (lo=r-1; lo>=0 && Double.isNaN(_data[lo][c+1]);--lo);
            // search for a number greater than x
            for (hi=r+1; hi<_rowCount && Double.isNaN(_data[hi][c+1]);++hi);
            if (lo < 0){
                // no data below x
                if (hi >=_rowCount) result = 0d; // no data in column at all
                result = _data[hi][c+1]; // this will be the lowest x value 
            }else if (hi >= _rowCount){
                // no data above x
                result = _data[lo][c+1]; // this will be the highest x value
            }else{
                // interpolate
                double xlo = _data[lo][0]; // column 0 is the x values
                double xhi = _data[hi][0];
                double ylo = _data[lo][c+1];
                double yhi = _data[hi][c+1];
                result = ylo + (yhi-ylo)*(x-xlo)/(xhi-xlo);
            } // if lo
        } // if isNan
        return result;
    } // getYAt

    public double getLowestX (int c){
        waitForWriters();
        for (int r=0; r<_rowCount; ++r){
            if (!Double.isNaN(_data[r][c+1])){
                return _data[r][0];
            } // if
        } // for
        return Double.POSITIVE_INFINITY;
    } // getLowestX

    public double getHighestX (int c){
        waitForWriters();
        for (int r=_rowCount-1; r>=0; --r){
            if (!Double.isNaN(_data[r][c+1])){
                return _data[r][0];
            } // if
        } // for
         return Double.NEGATIVE_INFINITY;
    } // getLowestX

    public double getNthX (int c, int n){
        if (n < 0) throw new IllegalArgumentException();
        waitForWriters();
        double[] point = new double[2];
        getNthPointWithData (c+1, n, point);
        return point[0];
    } // getNthX

    public double getNextX(int c, double x){
        if (x >= getHighestX(c)) return Double.POSITIVE_INFINITY;
        double result;
        waitForWriters();
        int r = findRow (x); // find the row >= t
        // if we found the original row, then go to the next one
        if (_data[r][0]==x) ++r;
        // now scan forward until we find a value
        while (Double.isNaN(_data[r][0])) ++r;
        return _data[r][0];
    } // getNextX
    
    public double getCount (int c){
        waitForWriters();
        return countData(c+1);
    } // getCount

    public int getColumnNumber(String name){
        waitForWriters();
        int result = findColumn(name);
        if (result==-1) throw new IllegalArgumentException(name+ "is not a column");
        return result;       
    }
    
    /** Creates a {@link util.DoubleBean} that reflects a given data point
     *  @param x the x value
     *  @param c the column
     *  @returns the DoubleBean
     *  @throws ArrayIndexOutOfBoundsException if the column does not exist
     **/
    public util.DoubleBean doubleBean(final double x, final int c){
        return new util.DoubleBean(){
            public double getValue() {return getY(x,c);}
            public void setValue (double d) {setY(x,c,d);}
        };        
    } // doubleBean

    /** Checks whether a given column is hidden
     *  @param c the column to check
     *  @return true if the column is hidden
     *  @throws ArrayIndexOutOfBoundsException if the column does not exist
     **/
     public boolean isHidden (int c) throws InterruptedException {
        waitForWriters();
        return _hidden[c+1];
     } // isHidden

    /** a string representation of a subset of this table.
     *  The first column of the table is the row headers, which does not have a title
     *  @param cols the indices of the columns to select
     *  @return a tab and newline delimited table
     *  @throws ArrayIndexOutOfBoundsException if the columns do not exist
     */
    public String toString (int[] cols ){
        waitForWriters();
        StringBuffer result = new StringBuffer();
        // create the header
        for (int c=0; c<_columnCount; ++c)
          result.append("\t").append(_names[c+1]);
        result.append("\n");
        // create the data
        for (int r=0; r<_rowCount; ++r){
            boolean hasData= false;
            StringBuffer line = new StringBuffer();
            for (int c=0; c<_columnCount; ++c){
                line.append ("\t");
                double y = _data[r][c+1];
                if (!Double.isNaN(y)){
                    line.append (y);
                    hasData = true;
                } // if
            } // for c
            if (hasData) 
              result.append(_data[r][0]).append(line).append("\n");
        } // for r
        return result.toString();
    } // toString
    

    //*******
    // Writer methods
    // All methods look like:
    //      if (SwingUtilities.isEventDispatchThread()){
    //          do something;
    //      }else{
    //          invokeWriter(new Runnable(){
    //              public void run(){
    //                  call myself;
    //              } // run
    //          }); // invokeWriter
    //      } // if
    
    /** Set the y value of a given row and column. This
     *  does not allow stting the X-value column (column -1)
     *  but does allow setting hidden columns
     *  @param r the row number
     *  @param c the column number
     *  @param y the new value
     *  @throws ArrayIndexOutOfBoundsException if the row or column do not exist
     **/
    public void setYAt (final int r, final int c, final double y){
        if (c < 0) throw new ArrayIndexOutOfBoundsException();
        if (SwingUtilities.isEventDispatchThread()){
            _data[r][c+1] = y;
            fireTableCellUpdated(r, c+1);
        }else{
            invokeWriter(new Runnable(){
                public void run() {
                    setYAt(r,c,y);
                } // run
            }); // invokeWriter
        } // if
    } // setYAt

    public void setY (final double x, final int c, final double y){
        if (SwingUtilities.isEventDispatchThread()){
            int r = findRowWithInsert (x);
            if (c == -1) return;
            _data[r][c+1] = y;
            fireTableCellUpdated(r, c+1);
        }else{
            invokeWriter(new Runnable(){
                public void run(){
                    setY(x,c,y);
                } // run
            }); // invokeWriter
        } // if
    } // setY
    
    public void addColumn (final String name){
        if (SwingUtilities.isEventDispatchThread()){
            int c = findColumn(name);
            if (c == -1) {
                appendColumn(name);
            }else{
                eraseColumn(c+1);
            } // if
            fireTableStructureChanged();
        }else{
            invokeWriter(new Runnable(){
                public void run(){
                    addColumn(name);
                } // run
            }); // invokeWriter
        } // if         
    } // addColumn

    /** Removes all blank rows
     */
    public void purgeRows (){
        if (SwingUtilities.isEventDispatchThread()){
            purgeRowsRaw();
            fireTableStructureChanged();
        }else{
            invokeWriter(new Runnable(){
                public void run(){
                    purgeRows();
                } // run
            }); // invokeWriter
        } // if         
    } // purgeRows

    /** Removes a column
     *  @param c the column to remove
     */
    public void removeColumn (final int c){
        if (SwingUtilities.isEventDispatchThread()){
            removeColumnRaw (c+1);
            fireTableStructureChanged();
        }else{
            invokeWriter(new Runnable(){
                public void run(){
                    removeColumn(c);
                } // run
            }); // invokeWriter
        } // if         
    } // addColumn
 
    /** Sets the hidden state for a column
     *  @param c the column
     *  @param hidden true to hide, false to show and edit.
     *  @throws ArrayIndexOutOfBoundsException if the row or column do not exist
     */
    public void setHidden (final int c, final boolean hide){
        if (SwingUtilities.isEventDispatchThread()){
            _hidden[c+1] = hide;
            fireTableDataChanged();
        }else{
            invokeWriter(new Runnable(){
                public void run(){
                    setHidden(c, hide);
                } // run
            }); // invokeWriter
        } // if         
    } // setHidden

    //*******
    // Locking methods
    protected void waitForWriters(){
        if (SwingUtilities.isEventDispatchThread()) return;
        try{
            SwingUtilities.invokeAndWait(new Runnable(){
                public void run(){}
            });
        }catch (InvocationTargetException ex){ /* do nothing */
        }catch (InterruptedException ex){ Thread.currentThread().interrupt(); }
    } // waitForWriters
 
    protected void invokeWriter (final Runnable r){
        SwingUtilities.invokeLater(r);
    } // invokeWriter

    //*******
    // Private methods.
    // All these assume that the appropriate locks have been acquired.
    // They also assume that the column numbers have been adjusted to
    // 1-based (the x values are now in column 0).
    
    // returns the smallest row with x-value >= x. If x > largest x-value,
    // return _rowCount.
    protected int findRow (double x){
        if (_rowCount < 1 || _data[_rowCount-1][0] < x){
            return (_lastRowSought = _rowCount);
        } // if
        if (_lastRowSought >= _rowCount) _lastRowSought = 0;
        return findRow (x, 0, _lastRowSought, _rowCount-1);
    } // findRow
    
    // does findRow but between min and max inclusive, with start as the
    // best guess
    protected int findRow (double x, int min, int start, int max){
        if (_data[start][0] == x){
            return (_lastRowSought = start);
        }else if (_data[min][0] >= x){
            return (_lastRowSought = min);           
        }else if (_data[max][0] == x){
            return (_lastRowSought = max);           
        }else if (_data[max][0] < x){
            return (_lastRowSought = max+1); 
        }else if (_data[start][0] < x){
            return findRow (x, start+1, (start+max)/2, max-1);
        }else{ // _data[start][0] > x
            return findRow (x, min+1, (min+start)/2, start-1);
        }
    } // findRow
    
    // does findRow but inserts the row if it does not exist
    protected int findRowWithInsert (double x){
        int r = findRow (x);
        if (r >= _rowCount || _data[r][0] != x){
            addRow (r);
            _data[r][0]=x;
            fireTableRowsInserted(r,r);
        }
        return r;
    } // findRowWithInsert
    
    // inserts a new row at r. This breaks the invariant
    // since _data[r][0] is left unset
    private void addRow (int r){
        // make sure we have enough room
        if (_rowCount >= _rowSize){
            double[][] temp = new double[_rowSize*2][];
            for (int row=0; row<_rowCount; ++row) temp[row] = _data[row];
            _data = temp;
            _rowSize = _data.length;
        } // if
        // move all the higher-up rows
        for (int row=_rowCount-1; row>=r; --row) _data[row+1]=_data[row];
        ++_rowCount;
        // insert the new row with no data
        _data[r]= new double[_columnSize];
        java.util.Arrays.fill (_data[r], Double.NaN);
        _rowHeaders.fireTableRowsInserted(r,r);
    } // addRow
    
    // adds a new column to the end of the column list. This does not
    // check that the column name does not already exist
    private void appendColumn (String name){
        // make sure we have enough room 
        // note that _columnCount does not include the x values column
        if (_columnCount+2 > _columnSize){
            _columnSize = _columnSize*2+1;
            String[] tempNames = new String[_columnSize];
            boolean[] tempHidden = new boolean[_columnSize];
            double[][] tempData = new double[_rowSize][_columnSize];
            for (int c=0; c<=_columnCount; ++c){
                tempNames[c] = _names[c];
                tempHidden[c] = _hidden[c];
                for (int r=0; r<_rowCount; ++r){
                    tempData[r][c] = _data[r][c];
                } // for r
            } // for c
            _names = tempNames;
            _hidden = tempHidden;
            _data = tempData;
        } // if
        ++_columnCount;
        _names[_columnCount] = name;
        _hidden[_columnCount] = false;
        for (int r=0; r<_rowCount; ++r){
            _data[r][_columnCount] = Double.NaN;
        } // for r
    } // appendColumn
    
    // Remove all data from a column
    protected void eraseColumn (int c){
        if (c<0) throw new IllegalArgumentException();
        for (int r=0; r<_rowCount; ++r) _data[r][c]=Double.NaN;
        fireTableDataChanged();
    } // eraseColumn
    
    // Remove all blank rows
    protected void purgeRowsRaw(){
        for (int r = _rowCount-1; r >= 0; --r){
            boolean hasData = false;
            for (int c = 1; c <= _columnCount; ++c){
                if (!Double.isNaN(_data[r][c])){
                    hasData = true;
                    break;
                } // if
            } // for c
            if (!hasData) removeRow (r);
        } // for r
    } // removeBlankRows
    
    // Remove the row r
    protected void removeRow (int r){
        // move all the higher-ups back
        for (int row=r; row<_rowCount-1; ++row) _data[row]=_data[row+1];
        --_rowCount;
    } // removeRow
    
    // Remove the column c
    protected void removeColumnRaw (int c){
        if (c < 1) throw new IllegalArgumentException();
        // move all the higher-ups back
        for (int col=c; col<_columnCount-1; ++col){
            _names[col]=_names[col+1];
            _hidden[col]=_hidden[col+1];
            for (int r = 0; r < _rowCount; ++r) _data[r][col]=_data[r][col+1];
        } // for
        --_columnCount;
    } // removeRow

    // Counts the number of data points in column c
    protected int countData (int c){
        int result = 0;
        for (int r=0; r<_rowCount; ++r) if (!Double.isNaN(_data[r][c])) ++result;
        return result;
    } // countData
    
    // returns the x and y values for the nth point that is not
    // NaN in a given column. Point must be at least double[2].
    protected void getNthPointWithData (int c, int n, double[] point){
        point[0] = Double.NaN;
        for (int r=0; r<_rowCount; ++r){
            if (!Double.isNaN(_data[r][c])){
                if (--n < 0){
                    point[0] = _data[r][0];
                    point[1] = _data[r][c];
                    break;
                } // if n
            } // if !isNaN
        } // for
        if (point[0] == Double.NaN) throw new IllegalArgumentException();
    } // getNthPointWithData

    //*******
    // Inner classes
    
    // TableModel that exposes the row headers (column "-1")
    class DataTableHeaders extends AbstractTableModel {
        public Class getColumnClass(int c) {return Double.class;}
        public int getColumnCount(){return 1;}
        public int getRowCount()
          {return DataTableImpl.this.getRowCount();}
        public Object getValueAt (int r, int c)
          {return DataTableImpl.this.getValueAt (r,-1);}
    } // DataTableHeaders
    
}  // DataTable
