/* ReactionBlock.java
 * created 15 July 2004
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.simulator;
import tenua.symbol.*;
import java.util.Iterator;
import java.util.List;


/** Analyzes a set of reactions.
 *  The reactions are defined by the right hand and left hand
 *  sides, each of which is a List, with the even numbered elements the stoichiometry
 *  and the odd elements a Symbol that represents the concentration of the species
 */
public class ReactionBlock {
    final Symbol[] _species;
    // Map to easily look up species numbers from their names
    final java.util.Map _speciesLookup = new java.util.HashMap();
    final SymbolTable _st;
    // arrays of stoichiometries, indexes as [reaction] [species]
    final int[][] _forward;
    final int[][] _backward;

    /** creates a ReactionBlock */
    public ReactionBlock (List lhs, List rhs, SymbolTable st){
        _st = st;
        // go through all the reactions and accumulate all the species
        java.util.Set speciesSet = new java.util.HashSet();
        for (Iterator i = lhs.iterator(); i.hasNext();){
            for (Iterator j = ((List)i.next()).iterator(); j.hasNext();){
                Object o = j.next();
                if (o instanceof Symbol) speciesSet.add(o);
            } // for j
        } // for i
        for (Iterator i = rhs.iterator(); i.hasNext();){
            for (Iterator j = ((List)i.next()).iterator(); j.hasNext();){
                Object o = j.next();
                if (o instanceof Symbol) speciesSet.add(o);
            } // for j
        } // for i
        _species = new Symbol [speciesSet.size()];
        speciesSet.toArray (_species);
        // create the lookup table
        for (int i = 0; i < _species.length; i++){
            _speciesLookup.put (_st.lookup(_species[i]), new Integer(i));
        } // for
        // accumulate the stoichiometries
        _forward = new int [rhs.size()] [_species.length];
        _backward = new int [rhs.size()] [_species.length];
        int reaction = 0;
        for (Iterator i = lhs.iterator(); i.hasNext(); reaction++){
            for (Iterator j = ((List)i.next()).iterator(); j.hasNext();){
                int stoich = ((Integer)j.next()).intValue();
                Symbol item = (Symbol) j.next(); // suck up two items at a time
                int species = ((Integer)_speciesLookup.get (_st.lookup(item))).intValue();
                _forward [reaction] [species] += stoich;
            } // for j
        } // for i
        reaction = 0;
        for (Iterator i = rhs.iterator(); i.hasNext(); reaction++){
            for (Iterator j = ((List)i.next()).iterator(); j.hasNext();){
                int stoich = ((Integer)j.next()).intValue();
                Symbol item = (Symbol) j.next(); // suck up two items at a time
                int species = ((Integer)_speciesLookup.get (_st.lookup(item))).intValue();
                _backward [reaction] [species] += stoich;
            } // for j
        } // for i

    } // constructor

    /** Creates an Expression for calculating a reaction rate.
     *  Calculated as
     *  <pre>
     *    rate(i)=k(i) * product_over_all_species_s (concentration[s]^stoichiometry[s])
     *  </pre>
     *  where stoichiometry is the stoichiometry of that species in that reaction in the
     *  appropriate direction.
     *  @param i the number of the reaction. use positive i for
     *  the forward reaction, negative i for the backward reaction. Note that
     *  reactions are numbered from 1.
     *  @return the Expression created
     *  @throws IllegalArgumentException if no such reaction exists
     */
    public Expression reactionRate (int i){
        if (i == 0 || i > _forward.length) throw new IllegalArgumentException();
        String reactionName = i>0 ? ("+" + i) : ("-" + -i);
        String rateConstant = "k("+reactionName+")";
        String rate = "rate("+reactionName+")";
        Expression result = new Expression (_st.get(rateConstant));
        for (int s = 0; s < _species.length; s++){
            int stoich = i > 0 ? _forward[i-1][s] : _backward[-i-1][s];
            if (stoich != 0){
                result.add (_species[s]);
                if (stoich > 1){
                    // append the exponent
                    result.add (Power.getInstance(stoich));
                } // if >1
                result.add (Binary.MUL);
            } // if !=0
        } // for
        result.add (assignment (rate));
        return result;
    } // reactionRate

    /** Creates an Expression that calculates the total rate of each reaction.
     *  Calculated as
     *  <pre>
     *    totalRate(r) = rate(+r)-rate(-r)
     *  </pre>
     *  @return the Expression created
     */
    public Expression totalRates(){
        Expression result = new Expression();
        for (int r = 1; r <= _forward.length; r++){
            String forwardRate = "rate(+"+r+")";
            String backwardRate = "rate(-"+r+")";
            String rateName = "totalRate("+r+")";
            result.add(_st.get(forwardRate));
            result.add(_st.get(backwardRate));
            result.add(Binary.SUB);
            result.add (assignment (rateName));
        } // for
        return result;
    } // totalRates

    /** Creates an Expression for calculating the rate of concentration increase
     *  for a chemical species.
     *  Calculated as
     *  <pre>
     *    rate(name) = sum_over_all_reactions_r ((backward_stoich[r] -forward_stoich[r])*totalRate(r))
     *  </pre>
     *  If the species is not mentioned in any of the reactions, the rate is zero
     *  @param name the name of the species
     *  @return the Expression created
     */
    public Expression speciesRate (String name){
        // using a starting value of zero makes coding easier (since
        // all the terms, even the first, will be added to something, though
        // the evaluation will be less efficient (since every evaluation pushes
        // zero onto the stack
        Expression result = new Expression(Value.ZERO);
        if (_speciesLookup.get(name) ==  null) return result;
        int s = ((Integer)_speciesLookup.get (name)).intValue();
        String rateName = "rate("+name+")";
        for (int r = 1; r <= _forward.length; r++){
            int stoich = _backward[r-1][s]-_forward[r-1][s];
            if (stoich == 0) continue;
            result.add (_st.get("totalRate("+r+")"));
            switch (stoich){
                case 1: 
                    result.add (Binary.ADD);
                    break;
                case -1:
                    result.add (Binary.SUB);
                    break;
                default:
                    result.add (_st.put(stoich)).add (Binary.MUL);
                    result.add (Binary.ADD);
            } // switch
        }  // for

        result.add (assignment (rateName));
        return result;
    } // speciesRate
    
    // returns a Symbol representing an assignment to the name's variable
    private Symbol assignment (String name){
        return new Expression(_st.put(name)).getAssignment(_st);
    } // assignment
} // ReactionBlock