/*
 * Binary.java
 *
 * Created on July 1, 2004, 10:26 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.symbol;
import java.util.Stack;

/** {@link Symbol} subclass that represents a binary operator.
 *  allows for precedence and associativity
 *  <br>
 *  Also exposes a large number of standard operators
 *
 * @author  Daniel Wachsstock
 */
public abstract class Binary extends Symbol {
    
    /** Creates a new instance of Binary */
    protected Binary() {
    }
    
    /** the precedence of this operator */
    abstract public int precedence();
    
    /** the associativity of this operator
     *  @return true if it is left-associative,
     *  false if right-associative
     */
    abstract public boolean leftAssociative();
    
    public void toString(Stack stringStack, Stack precedenceStack, SymbolTable st) {
            StringBuffer rhs = (StringBuffer) stringStack.pop();
            int rhsPrecedence = ((Integer) precedenceStack.pop()).intValue();
            StringBuffer lhs = (StringBuffer) stringStack.pop();
            int lhsPrecedence = ((Integer) precedenceStack.pop()).intValue();
            int precedence = precedence();
            boolean leftAssociated = leftAssociative();
            if (rhsPrecedence < precedence ||
              (leftAssociated && rhsPrecedence == precedence))
              parenthesize(rhs);
            if (lhsPrecedence < precedence ||
              (!leftAssociated && lhsPrecedence == precedence))
              parenthesize(lhs);
            lhs.append(this).append(rhs);
            stringStack.push(lhs);
            precedenceStack.push(new Integer(precedence));
    } // toString
    
    // static final operators that can be shared
    static public final Binary ADD = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            s.push (s.pop() + s.pop());
        }
        public int precedence() {return 4;}
        public boolean leftAssociative() {return true;}
        public String toString(){ return "+";}
    };
    static public final Binary SUB = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            s.push (s.pop() - rhs);
        }
        public int precedence() {return 4;}
        public boolean leftAssociative() {return true;}
        public String toString(){ return "-";}
    };
    static public final Binary MUL = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            s.push (s.pop() * s.pop());
        }
        public int precedence() {return 5;}
        public boolean leftAssociative() {return true;}
        public String toString(){ return "*";}
    };
    static public final Binary DIV = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            s.push (s.pop() / rhs);
        }
        public int precedence() {return 5;}
        public boolean leftAssociative() {return true;}
        public String toString(){ return "/";}
    };
    static public final Binary MOD = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            s.push (s.pop() % rhs);
        }
        public int precedence() {return 5;}
        public boolean leftAssociative() {return true;}
        public String toString(){ return "%";}
    };
    static public final Binary POW = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            s.push (Math.pow (s.pop(), rhs));
        }
        public int precedence() {return 6;}
        public boolean leftAssociative() {return false;}
        public String toString() { return "^"; }
    };
    static public final Binary AND = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs!=0 && rhs!=0 ? 1d : 0d);
        }
        public int precedence() {return 2;}
        public boolean leftAssociative() {return true;}
        public String toString() { return "&"; }
    };
    static public final Binary OR = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs!=0 || rhs!=0 ? 1d : 0d);
        }
        public int precedence() {return 2;}
        public boolean leftAssociative() {return true;}
        public String toString() { return "|"; }
    };
    static public final Binary EQ = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs == rhs? 1d : 0d);
        }
        public int precedence() {return 3;}
        public boolean leftAssociative() {return true;}
        public String toString() { return "=="; }
    };
    static public final Binary NE = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs != rhs? 1d : 0d);
        }
        public int precedence() {return 3;}
        public boolean leftAssociative() {return true;}
        public String toString() { return "!="; }
    };
    static public final Binary LE = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs <= rhs? 1d : 0d);
        }
        public int precedence() {return 3;}
        public boolean leftAssociative() {return true;}
        public String toString() { return "<="; }
    };
    static public final Binary GE = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs >= rhs? 1d : 0d);
        }
        public int precedence() {return 3;}
        public boolean leftAssociative() {return true;}
        public String toString() { return ">="; }
    };
    static public final Binary LT = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs < rhs? 1d : 0d);
        }
        public int precedence() {return 3;}
        public boolean leftAssociative() {return false;}
        public String toString() { return "<"; }
    };
    static public final Binary GT = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs > rhs? 1d : 0d);
        }
        public int precedence() {return 3;}
        public boolean leftAssociative() {return true;}
        public String toString() { return ">"; }
    };
    static public final Binary IF = new Binary(){
        public void eval (DoubleStack s, VariableMemento v){
            double rhs = s.pop();
            double lhs = s.pop();
            s.push (lhs != 0d ? rhs : 0d);
        }
        public int precedence() {return 1;}
        public boolean leftAssociative() {return false;}
        public String toString() { return "?"; }
    };

}
