/* Expression--a sequence of Symbols that can be evaluated
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.symbol;

import java.util.*;

/** A sequence of {@link Symbol}s that are evaluated in order, like an RPN calculator */
public class Expression{
    private final Vector _symbols = new Vector();

    /** create a new empty Expression*/
    public Expression(){
    }

    /** create a new Expression that contains a single {@link Symbol} */
    public Expression (Symbol s){
        _symbols.add(s);
    }
    
    /** add a {@link Symbol} to the end of this
     *  @param s the Symbol to add
     *  @return this
     */
    public Expression add (Symbol s){
        _symbols.add(s);
        return this;
    } // add
    
    /** add all the contents of an existing Expression to the end of this
     *  @param e the Expression to add to this
     *  @return this
     */
    public Expression add (Expression e){
        _symbols.addAll(e._symbols);
        return this;
    } // add
    
    /** insert a Symbol to the start of this
     *  @param s the Symbol to add
     *  @return this
     */
    public Expression prepend (Symbol s){
        _symbols.add(0,s);
        return this;
    } // prepend

    /** insert all the contents of an existing Expression to the start of this
     *  @param e the Expression to add
     *  @return this
     */
    public Expression prepend (Expression e){
        _symbols.addAll(0,e._symbols);
        return this;
    } // prepend
    
    /** returns an {@link Assignment} representing an assignment operation to this.
     *  valid only if this is a single Symbol which is a variable
     *  @param st the SymbolTable to use for error messages (using null will use
     *  cryptic names (like "Variable[0]") but thhe method will work
     *  @return the Assignment created
     *  @throws UnexpectedSymbolException if this cannot be assigned to
     */
    public Assignment getAssignment (SymbolTable st){
        if (_symbols.size() == 1 && _symbols.get(0) instanceof Variable){
            return new Assignment((Variable) _symbols.get(0));
        }else{
            throw UnexpectedSymbolException.instance
              (toString(st), "Symbol.CannotAssign");            
        }
    } // getAssignment

    /** evaluate this
     *  @param v the memento representing the values of the variables.
     *  if v == null, use the default values
     *  @return a copy of the values from the stack
     */
    public nr.Vec eval (VariableMemento v){
        DoubleStack stack = new DoubleStack();
        Iterator i;
        for (i=_symbols.iterator(); i.hasNext();){
            ((Symbol) i.next()).eval(stack, v);
        }
        return stack.getValues();
    } // eval
    
    /** evaluate this
     *  @param d the array of values to push onto the stack before evaluating.
     *  Pushed in reverse order (the stack is created with {@link DoubleStack#DoubleStack(nr.Vec)}.
     *  @param v the memento representing the values of the variables.
     *  if v == null, use the default values
     *  @return a copy of the values from the stack
     */
    public nr.Vec eval (nr.Vec d, VariableMemento v){
        DoubleStack stack = new DoubleStack(d);
        Iterator i;
        for (i=_symbols.iterator(); i.hasNext();){
            ((Symbol) i.next()).eval(stack, v);
        }
        return stack.getValues();
    } // eval
    
     /** evaluate this
     *  @param d the array of values to push onto the stack before evaluating.
     *  Pushed in reverse order (the stack is created with {@link DoubleStack#DoubleStack(nr.Vec)}.
     *  @param result a copy of the values from the stack after evaluation
     *  @param v the memento representing the values of the variables.
     *  if v == null, use the default values
     */
    public void eval (nr.Vec d, nr.Vec result, VariableMemento v){
        DoubleStack stack = new DoubleStack(d);
        Iterator i;
        for (i=_symbols.iterator(); i.hasNext();){
            ((Symbol) i.next()).eval(stack, v);
        }
        result.set(stack.getValues());
    } // eval

    /** produces an RPN string, with variables listed as Variable[n].
     *  for debugging; not for showing to users
     *  @return the String produced
     */
    public String toString(){
        StringBuffer name = new StringBuffer();
        for (Iterator i=_symbols.iterator(); i.hasNext();){
            if (name.length() > 0) name.append(" ");
            name.append(i.next());
        } // for
        return name.toString();
    } // toString
        
    /** produces a String representation of this expression.
     *  Uses the names from a {@link SymbolTable}.
     *  parenthesizes only what is necessary by the rules of precedence. 
     *  This is rather kludgy, but it seems to work
     *  @param st the {@link SymbolTable} to use. If null, uses {@link #toString()}
     *  @return the String produced
     */
    public String toString (SymbolTable st){
        if (st==null) return toString();
        Stack stringStack = new Stack();
        Stack precedenceStack = new Stack();
        for (Iterator i=_symbols.iterator(); i.hasNext();){
            Object s = i.next();
            if (s instanceof Symbol)
              ((Symbol) s).toString(stringStack, precedenceStack, st);
        } // for
        StringBuffer result = new StringBuffer();
        while (!stringStack.isEmpty()){
            if (result.length() > 0) result.insert(0,';');
            result.insert(0,stringStack.pop());
        }
        return result.toString();
    } // toString

    /** tests whether an expression is constant (no variables)
     *  @return true if this is constant,
     *  false if not.
     */
    public boolean isConstant(){
        for (Iterator i=_symbols.iterator(); i.hasNext();){
            Object s = i.next();
            if (s instanceof Variable) return false;
        } // for
        return true;
    } // isConstant
    
} // Expression
