/*
 * Power.java
 *
 * Created on July 2, 2004, 1:31 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.symbol;
import java.util.Stack;

/** A {@link Symbol} that represents an integer
 *  power
 *
 * @author  Daniel Wachsstock
 */
public class Power extends Symbol {
    private int _i;
    
    // All creation should go through getInstance
    private Power (int i) {
        super();
        _i = i;
    }
    
    static public Power getInstance (int i){
        if (i < 0) {
            final Power posPower = getInstance(-i);
            return new Power(i){
                Power _posPower = posPower;
                public void eval (DoubleStack s, VariableMemento v){
                    posPower.eval(s,v);
                    s.push (1d / s.pop());
                } // eval
            }; // return new Power
        }else switch (i){
            case 0: return ZERO;
            case 1: return ONE;
            case 2: return TWO;
            case 3: return THREE;
            case 4: return FOUR;
            case 5: return FIVE;
            case 6: return SIX;
            case 7: return SEVEN;
            // we could keep a list of already allocated
            // Power's and return those, but that seems like more
            // work than is warranted
            default: return new Power(i);
        } // if
    } // getInstance

    // semi-fancy algorithm for integer powers that calculates
    // val^n as a running product of squares of val, but only
    // multiplying the sqaure into the running product if the
    // corresponding bit of n is set.
    // so val^10 = val^1010 in base two,
    // so result =  val(1*2) * val^(4*2) (2nd and 4th bits of n
    // are set). The successive squares of val are calculated by
    // squaring val each time through the loop, and the bits
    // are calculated by shifting and &'ing with 1.
    public void eval (DoubleStack s, VariableMemento  v){
        double val = s.pop();
        double result = 1.0d;
        for (int n = _i; n > 0; n >>= 1, val *= val){
            if ((n & 1) == 1) result *= val;
        } // for
        s.push (result);
    } // eval   

    public String toString(){ return "^"+_i; }

    // use the precedence and associativity from Binary.POW
    public void toString(Stack stringStack, Stack precedenceStack, SymbolTable st) {
        StringBuffer top = (StringBuffer) stringStack.peek(); // modify in place
        int topPrecedence = ((Integer) precedenceStack.pop()).intValue();
        int precedence = Binary.POW.precedence();
        boolean leftAssociated = Binary.POW.leftAssociative();
        if (topPrecedence < precedence ||
          (!leftAssociated && topPrecedence == precedence))
          parenthesize(top);
        top.append(this);
        precedenceStack.push(new Integer(precedence));
    }
    
    // static finals that can be shared
    // these are the ones for which a simple loop was faster than
    // the shift-mulitply method
    static public final Power ZERO = new Power(0){
        public void eval (DoubleStack s, VariableMemento v){
            s.pop();
            s.push (1d);
        }
    };
    static public final Power ONE = new Power(1){
        public void eval (DoubleStack s, VariableMemento v){
            /* do nothing */
        }
    };
    static public final Power TWO = new Power(2){
        public void eval (DoubleStack s, VariableMemento v){
            double val = s.pop();
            s.push (val*val);
        }
   };
    static public final Power THREE = new Power(3){
        public void eval (DoubleStack s, VariableMemento v){
            double val = s.pop();
            s.push (val*val*val);
        }
   };
    static public final Power FOUR = new Power(4){
        public void eval (DoubleStack s, VariableMemento v){
            double val = s.pop();
            val *= val;
            s.push (val*val);
        }
   };
    static public final Power FIVE = new Power(5){
        public void eval (DoubleStack s, VariableMemento v){
            double val = s.pop();
            double valSquared = val * val;
            s.push (val*valSquared);
        }
   };
   static public final Power SIX = new Power(6){
        public void eval (DoubleStack s, VariableMemento v){
            double val = s.pop();
            val *= val;
            s.push (val*val*val);
        }
    };
    static public final Power SEVEN = new Power(7){
        public void eval (DoubleStack s, VariableMemento v){
            double val = s.pop();
            double valSquared = val*val;
            s.push (valSquared*valSquared*val);
        }
   };
    
}
