/* SymbolTable--a Hashtable of String => Symbol
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */
package tenua.symbol;

import java.util.*;

public class SymbolTable{
    private final Hashtable _t = new Hashtable(); // threadsafe
    private int _numVariables = 0; // the number of variables created so far
    // the preferences node to save the default initial values
    private static final java.util.prefs.Preferences _prefs =
      java.util.prefs.Preferences.userNodeForPackage(SymbolTable.class);

    /** returns the Symbol associated with key */
    public Symbol get (String key){
        return (Symbol) _t.get(key);
    }

    /** inserts n in the SymbolTable associated with key, replacing
     *  the previous association
     *  @param key the name
     *  @param s the symbol
     *  @return the new value in this SymbolTable of key
     */
    public Symbol put (String key, Symbol s){
        _t.put(key, s);
        return s;
    }

    /** adds an unnamed constant to the SymbolTable 
     *  @param d the value to add
     *  @return the Symbol created
     *  @throws UnexpectedSymbolException if a Symbol with the name
     *  <code>Double.toString(d)</code> already exists but is not a Value
     */
    public Symbol put (double d) throws UnexpectedSymbolException{
        Symbol n = get (Double.toString(d));
        if (n != null){
            if ( n instanceof Value){
                return n;
            }else{ // should be a number, but ain't
                throw UnexpectedSymbolException.instance
                  (d+"", "Symbol.NotANumber");
            }
        }

        return put (Double.toString(d), new Value(d));
    } // put

    /** adds a named constant to the SymbolTable 
     *  @param name the name of the constant
     *  @param d its value
     *  @return the Symbol created
     *  @throws UnexpectedSymbolException if the name was previously defined
     */
    public Symbol put (String name, double d) {
        Symbol n = get (name);
        if (n != null){
            throw UnexpectedSymbolException.instance
              (name, "Symbol.Redefined");
        }

        return put (name, new Value(d));
    } // put

    /** adds a new variable 
     *  @param name the name of the new Symbol
     *  @return the Symbol created
     *  @throws UnexpectedSymbolException if name was previously defined
     *  to something that is not a variable or value
     */
    public Symbol put (String name){
        Symbol n = get(name);
        if (n != null){
            if (n instanceof Value){
                return n;
            }else{ // should be a value, but ain't
                throw UnexpectedSymbolException.instance
                  (name, "Symbol.NotAValue");
            }
        }

        Variable  v = new  Variable (_numVariables++);
        v.assign (_prefs.getDouble(name, 0d), null); // default to 0
        return put (name, v);
    } // put

    /** Looks for an Object in the symbol table
     *  @param s the Object to look for
     *  @return the name of the Object if found; null otherwise
     */
    public String lookup (Object s){
        for (Iterator i = _t.entrySet().iterator(); i.hasNext();){
            Map.Entry item = (Map.Entry) i.next();
            if (item.getValue().equals(s)) return (String) item.getKey();
        } // for
        return null;
    } // lookup
    
    /** returns a memento of all the variable default values */
    public VariableMemento getMemento(){
        VariableMemento v = new VariableMemento(_numVariables);
        for (Iterator i = _t.entrySet().iterator(); i.hasNext();){
            Map.Entry item = (Map.Entry) i.next();
            if (! (item.getValue() instanceof Variable)) continue;
            Variable var = (Variable) item.getValue();
            v._d[var._i] = var._d;
        } // for
        return v;
    } // getMemento
    
    /** returns a Vec that represents a subset of the values in the symbol table
     *  @param names List of String of the names of the symbols
     *  @param v the VariableMemento to use
     *  @return the Vec. Vec.get(i) will the same as get(name[i]).getValue(v)
     *  and Vec.set(i,d)
     *  will be the same as ((Variable) get(name[i])).assign (v, d)
     *  @throws UnexpectedSymbolException if a name is not a Value
     */
    public nr.Vec subset (List names, VariableMemento v){
        return new Subset(names, v);
    } // subset
    
    /** returns a Vec that represents a subset of the values in the symbol table
     *  @param names String[] of the names of the symbols
     *  @param v the VariableMemento to use
     *  @return the Vec. Vec.get(i) will the same as get(name[i]).getValue(v)
     *  and Vec.set(i,d)
     *  will be the same as ((Variable) get(name[i])).assign (v, d)
     *  @throws UnexpectedSymbolException if a name is not a Value
     */
    public nr.Vec subset (String[] names, VariableMemento v){
        return new Subset(names, v);
    } // subset
    
    /** returns an array of names of all the variables */
    public String[] variableList(){
        String[] result = new String[_numVariables];
        int count = 0;
        for (Iterator i = _t.entrySet().iterator(); i.hasNext();){
            Map.Entry item = (Map.Entry) i.next();
            if (! (item.getValue() instanceof Variable)) continue;
            result[count++] = (String) item.getKey();
        } // for
        return result;       
    } // variableList
    
    /** Saves a default value to the user preferences if one was not
     *  previously saved
     *  @param name the name of the variable to save
     *  @param val the value
     */
    static public void ensureDefault (String name, double val){
        double d = _prefs.getDouble (name, val);
        // if we got back our default, then it likely didn't exist and we
        // should save it
        if (d == val) _prefs.putDouble (name, d);
    } // ensureDefault
    
    /** Saves all the current values of the variables into the user preferences
     *  @param v the VariableMemento to use (if null, uses the default values)
     */
    public void saveToPreferences (VariableMemento v){
        for (Iterator i = _t.entrySet().iterator(); i.hasNext();){
            Map.Entry item = (Map.Entry) i.next();
            if (! (item.getValue() instanceof Variable)) continue;
            _prefs.putDouble ((String) item.getKey(),
              ((Variable) item.getValue()).getValue(v));
        } // for        
    } // saveToPreferences

    class Subset implements nr.Vec {
        final private Value[] _s;
        final private VariableMemento _v;

        Subset (List names, VariableMemento v) throws UnexpectedSymbolException{
            _v = v;
            if (names == null){
                _s = new Value[0];
                return;
            } // if
            _s = new Value[names.size()];
            for (int i = 0; i < _s.length; i++){
                Symbol sym = SymbolTable.this.get((String)names.get(i));
                if (sym == null || ! (sym instanceof Value))
                    throw UnexpectedSymbolException.instance
                      (names.get(i).toString(), "Symbol.NotAValue");
                _s[i] = (Value) sym;
            } // for
        } // constructor

        Subset (String[] names, VariableMemento v) throws UnexpectedSymbolException{
            _v = v;
            if (names == null){
                _s = new Value[0];
                return;
            } // if
            _s = new Value[names.length];
            for (int i = 0; i < _s.length; i++){
                Symbol sym = SymbolTable.this.get(names[i]);
                if (sym == null || ! (sym instanceof Value))
                    throw UnexpectedSymbolException.instance
                      (names[i].toString(), "Symbol.NotAValue");
                _s[i] = (Value) sym;
            } // for
        } // constructor
        
        public double[] asArray() {
            double[] result = new double[_s.length];
            for (int i = 0; i < _s.length; i++) result[i] = get(i);
            return result;
        } // asArray

        public nr.Vec copy() {
            return new nr.Vec_array(this);
        } // copy

        public double get(int i) {
            return _s[i].getValue(_v);
        }

        public void set(double[] d) {
            for (int i = 0; i < d.length; i++) set(i, d[i]);
        }

        public void set(nr.Vec v) {
            if (this == v) return;
            for (int i = 0; i < v.size(); i++) set(i, v.get(i));
        }

        public void set(int i, double d) {
            if (_s[i] instanceof Variable){
                ((Variable) _s[i]).assign(d, _v);
            }else{
                // not a variable; throw an error
                String name = lookup(_s[i]);
                if (name == null){
                    throw UnexpectedSymbolException.instance
                      (_s[i].toString(), "Symbol.Undefined");
                }else{
                    throw UnexpectedSymbolException.instance
                      (name, "Symbol.CannotBeSet");
                } // if name == null
            } // if _s[i]
        } // set

        public int size() {
            return _s.length;
        } // size
        
        public String toString(){
            String result = "{";
            for (int i = 0; i < _s.length; i++){
                if (i > 0) result += ", ";
                result += get(i);
            } // for
            return result+"}";
        }// toString

    } // Subset
    
} // SymbolTable


