/*
 * Unary.java
 *
 * Created on July 2, 2004, 1:31 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package tenua.symbol;
import java.util.Stack;

/** A {@link Symbol} that represents a unary operator.
 *  does nothing that a regular {@link Symbol} doesn't.
 *  but is useful for distinguishing types
 *
 * @author  Daniel Wachsstock
 */
abstract public class Unary extends Symbol {
    
    /** Creates a new instance of Unary */
    protected Unary() {
    }

    // true if the function is easier to read if its argument is parenthesized
    // e.g. sin(x) rather than sinx, so shouldParenthesize would be true
    // but -x is easier than -(x), so shouldParenthesize would be false
    abstract boolean shouldParenthesize();
    
    public void toString(Stack stringStack, Stack precedenceStack, SymbolTable st) {
        StringBuffer top = (StringBuffer) stringStack.peek(); // modify in place
        int topPrecedence = ((Integer) precedenceStack.pop()).intValue();
        if (shouldParenthesize() || topPrecedence < maxPrecedence.intValue())
          parenthesize(top);
        top.insert(0, this);
        precedenceStack.push (maxPrecedence);
    }
    
    // static finals that can be shared
    static public final Unary NEGATE = new Unary(){
        public void eval (DoubleStack s, VariableMemento v){
            s.push ( - s.pop());
        }
        public String toString(){ return "-";}
        boolean shouldParenthesize() {return false;}
    };
    static public final Unary NOT = new Unary(){
        public void eval (DoubleStack s, VariableMemento v){
            s.push (s.pop() == 0d ? 1d : 0d);
        }
        public String toString(){ return "!";}
        boolean shouldParenthesize() {return false;}
    };
    static public final Unary EXP = new Unary(){
        public void eval (DoubleStack s, VariableMemento v){
            s.push ((float)Math.exp (s.pop()));
        }
        public String toString(){ return "exp";}
        boolean shouldParenthesize() {return true;}
    };
    static public final Unary LN = new Unary(){
        public void eval (DoubleStack s, VariableMemento v){
            s.push ((float)Math.log (s.pop()));
        }
        public String toString(){ return "ln";}
        boolean shouldParenthesize() {return true;}
   };
    
}
