/*
 * DoubleBean.java
 *
 * Created on September 7, 2005, 5:19 PM
 *
 *  Copyright 2005 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
*/

package util;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.text.DecimalFormat;
import javax.swing.AbstractAction;
import javax.swing.Box;
import javax.swing.DefaultCellEditor;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

/** A "bean" (class with get/set methods) that records a double.
 *  Useful with the Beanshell syntactic sugar that turns
 *  <code>bean.value</code> into <code>bean.getValue()</code> and
 *  <code>bean.value=x</code> into <code>bean.setValue(x)</code>.
 *  Also includes a convenience methods to create a JTextfield,
 *  a labelled JTextfield, a TableCellRenderer and a TableCellEditor
 *  that  reflects the value.
 *
 * @author Daniel Wachsstock
 */
abstract public class DoubleBean {

    abstract public double getValue();
    abstract public void setValue (double x);

    protected static final int NUM_DIGITS = 3;
    protected static final DecimalFormat POINT_FORMAT =
        new DecimalFormat("0.#");
    protected static final DecimalFormat E_FORMAT =
        new DecimalFormat("0.#E0");

    /** the natural logarithm of 10, for calculating log base 10 */
    public static final double LOG10 = Math.log(10d);

    /** a String that is larger than anything produced by {@link #toString}
     */
    public static final String MAX_STRING = "-#.###E-###";
    
    /** the value as a String. Formats numbers to a string
     *  in a way that I like--limited significant digits,
     *  scientific notation if they are too large or small.
     *  Formats a NaN as " " (one space)
     *  Decimal Format is just too inflexible.
     *  @return the String
     */
    public String toString(){ return toString(getValue()); } 

    /** the value as a String. Formats numbers to a string
     *  in a way that I like--limited significant digits,
     *  scientific notation if they are too large or small.
     *  Formats a NaN as " " (one space).
     *  Decimal Format is just too inflexible.
     *  <br>
     *  This is synchronized, so multiple threads can use it.
     *  @param d the double to format
     *  @return the String
     */
    static public synchronized String toString (double d){
        if (d == 0) return "0";
        if (Double.isNaN(d)) return " ";
        StringBuffer result = new StringBuffer();
        double abs = Math.abs(d);
        DecimalFormat format;
        if (abs < 0.1d){
            format = E_FORMAT;
            format.setMaximumFractionDigits(NUM_DIGITS);
        }else if (abs < 1d){
            format = POINT_FORMAT;
             // allow for the first digit being 0
            format.setMaximumFractionDigits(NUM_DIGITS+1);
        }else{
            int dDigits = (int) (Math.log(abs)/LOG10);
            if (dDigits > NUM_DIGITS){
                format = E_FORMAT;
                format.setMaximumFractionDigits(NUM_DIGITS);
            }else{ // small enough to use fixed point
                format = POINT_FORMAT;
                format.setMaximumFractionDigits(NUM_DIGITS - dDigits);
                format.setDecimalSeparatorAlwaysShown(false);
            } // dDigits
        } // d
        return format.format(d).toString();
    } // toString    
    
    /** Create a Swing Box that reflects this value.
     *  It is composed of a JLabel and a JTextfield under that, with the
     *  value in the textbox.
     *  Editing the text box will call {@link setValue}
     *  but the reverse is not automatically true; if <code>setValue</code>
     *  is called, the text box does not change unless the subclass's
     *  <code>setValue</code> explicitly does so. However, each repaint
     *  of this component calls {@link getValue}, so calling
     *  <code>box.repaint()</code> after <code>setValue</code> will
     *  accomplish that.
     *  Imposes an input verifier such that, if the edited text does not
     *  represent a number, the text reverts to the original.
     *  @param label the initial label on the box
     *  @return the Box
     */
    public Box box (String label){
        if (label == null) label = "";
        final JLabel jlabel = new JLabel(label);
        final JTextField jtext = textfield();
        Box b = new Box(javax.swing.BoxLayout.Y_AXIS){
            public void setEnabled (boolean bool){
                jtext.setEditable (bool);
                super.setEnabled (bool);
            } // setEnabled
            public void setName (String name){
                jlabel.setText (name);
                super.setName(name);
            } // setName
        }; // new Box
        b.add(jlabel);
        b.add(jtext);
        jlabel.setLabelFor(jtext);
        // set the maximum size so we keep one-line text fields
        Dimension size = jlabel.getPreferredSize();
        size.height *= 2;
        size.width = Integer.MAX_VALUE;
        b.setMaximumSize(size);
        
        return b;
    } // box
    
    /** Create a Swing JTextField that reflects this value.
     *  Editing the text box will call {@link setValue}
     *  but the reverse is not automatically true; if <code>setValue</code>
     *  is called, the text box does not change unless the subclass's
     *  <code>setValue</code> explicitly does so. However, each repaint
     *  of this component calls {@link getValue}, so calling
     *  <code>text.repaint()</code> after <code>setValue</code> will
     *  accomplish that.
     *  Imposes an input verifier such that, if the edited text does not
     *  represent a number, the text reverts to the original.
     *  @return the JTextField
     */
    public JTextField textfield(){
        final JTextField jtext = new JTextField(){
            protected void paintComponent (java.awt.Graphics g){
                setText(DoubleBean.this.toString());
                super.paintComponent(g);
            } // paintComponent
        }; // new JTextField
        jtext.setInputVerifier(new NumberVerifier(this));
        jtext.getInputMap().put(KeyStroke.getKeyStroke("ENTER"),"enter");
        jtext.getActionMap().put("enter", new AbstractAction(){
            public void actionPerformed (ActionEvent e){
                jtext.getInputVerifier().shouldYieldFocus(jtext);
            } // actionPerformed
        }); // new Action
        return jtext;
    } // textfield

    /** create a table cell renderer that displays a number in the same format
     *  as a DoubleBean.textfield.
     *  @return the TableCellRenderer
     *  @throws NumberFormatException if the value of the cell is not a Number
     */
    static public TableCellRenderer cellRenderer(){
        return new DefaultTableCellRenderer() {        
            public Component getTableCellRendererComponent (JTable table, Object value,
              boolean isSelected, boolean hasFocus, int row, int column){
                // get the default component
                Component result = super.getTableCellRendererComponent (table, value,
                  isSelected, hasFocus, row, column);
                setText (DoubleBean.toString(((Number)value).doubleValue()));
                setHorizontalAlignment(RIGHT);
                if (isSelected) {
                    setForeground(table.getSelectionForeground());
                    setBackground(table.getSelectionBackground());
                }else{
                    setForeground(table.getForeground());
                    setBackground(table.getBackground());
                } // if isSelected
                if (!table.getModel().isCellEditable(row, column)){
                    setForeground(java.awt.Color.LIGHT_GRAY);
                } 
                return result;
            } // getTableCellRendererComponent
        }; // new DefaultTableCellRenderer
    } // cellRenderer

    /** create a cell editor that uses 
     *  a DoubleBean.textfield.
     *  @return the TableCellEditor. This is guaranteed to also implement
     *  TreeCellEditor, if you want to cast it that way.
     *  @throws NumberFormatException if the value of the cell is not a Number
     */
    public static TableCellEditor cellEditor(){
        final DoubleBean bean = new DoubleBean(){
            double _value = Double.NaN;
            public double getValue() {return _value;}
            public void setValue (double x) {_value=x;}
        }; // new DoubleBean
        // bean.textfield does not work; overriding paintComponent causes problems
        final JTextField jtext = new JTextField();
        jtext.setInputVerifier(new NumberVerifier(bean));
        return new DefaultCellEditor(jtext){
            public Component getTableCellEditorComponent
              (JTable table, Object value, boolean isSelected, int r, int c){
                JTextField text = (JTextField)super.getTableCellEditorComponent
                  (table, value, isSelected, r, c);
                bean.setValue (((Number)value).doubleValue());
                text.setText (bean.toString());
                text.setHorizontalAlignment(JTextField.TRAILING);
                text.select (0, Integer.MAX_VALUE);
                return text;
            } // getTableCellEditorComponent
            public Object getCellEditorValue(){
                try{
                    // %%% return Double.valueOf(((JTextField)getComponent()).getText());
                    return Double.valueOf (jtext.getText());
                }catch (NumberFormatException ex){
                    return new Double(bean.getValue());
                }
            } // getCellEditorValue
        }; // new DefaultCellEditor
    } // cellEditor

    static class NumberVerifier extends javax.swing.InputVerifier {
        double _d; // used to hold the last value tested
        // technically not threadsafe, but should only be used on the Swing
        // event thread.
        final DoubleBean _source;
        JTextField _text;
        
        public NumberVerifier (DoubleBean source) { _source = source; }
        
        public boolean verify(JComponent input) {
            _text = (JTextField)input;
            try{
                _d = Double.parseDouble(_text.getText());
                return true;
            }catch (NumberFormatException ex){
                _d = _source.getValue(); // reset the value to the original
                return false;
            } // try
        } // verify

        public boolean shouldYieldFocus(JComponent input){
            if (verify(input)){ 
                _source.setValue(_d);
            }else{
                _text.setText (toString());
            }
            return true; // always yield focus
        } // shouldYieldFocus
    } // NumberVerifier

} // ValueBean
