/*
 * Num2Str.java
 *
 * Created on July 8, 2004, 7:57 PM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package util;
import java.text.DecimalFormat;

/** a simple class to format numbers to a string
 *  in a way that I like--limited significant digits,
 *  scientific notation if they are too large or small.
 *  Decimal Format is just too inflexible
 *
 * @author  Daniel Wachsstock
 */
public class Num2Str {
    private int _numDigits = 3;
    private final DecimalFormat _format = new DecimalFormat();
    
    /** the natural logarithm of 10, for calculating log base 10 */
    public static final double LOG10 = Math.log(10d);
    
    /** a String that is larger than anything produced by {@link #format(double)}
     *  using the default number of digits.
     *  @see #maxString()
     */
    public static final String MAX_STRING = "-#.###E-###";
    
    /** Creates a new instance of Num2Str with 3 significant digits after the decimal */
    public Num2Str() {}
    
    /** Creates a new instance of Num2Str with a set number of
     *  significant digits.
     *  @param numDigits the number of significant digits after the decimal
     *  (numDigits + 1) total
     *  @throws IllegalArgumentException if numDigits <= 0
     */
    public Num2Str (int numDigits){
        if (numDigits <= 0) throw new IllegalArgumentException();
        _numDigits = numDigits;
    }
    
    /** format a number.
     *  For numbers with absolute value less than 0.1, use scientific notation.
     *  For numbers > 10^numdigits, use scientific notation
     *  For numbers in between, use fixed notation
     */
    public StringBuffer format(double d){
        if (d == 0) return new StringBuffer("0");
        StringBuffer result = new StringBuffer();
        if (d < 0){
            d = -d;
            result.append("-");
        }
        DecimalFormat format;
        if (d < 0.1d){
            format = new DecimalFormat("0.#E0");
            format.setMaximumFractionDigits(_numDigits);
        }else if (d < 1d){
            format = new DecimalFormat("0.#");
            format.setMaximumFractionDigits(_numDigits+1); // allow for the first digit being 0
        }else{
            int dDigits = (int) (Math.log(d)/LOG10);
            if (dDigits > _numDigits){
                format = new DecimalFormat ("0.#E0");
                format.setMaximumFractionDigits(_numDigits);
            }else{ // small enough to use fixed point
                format = new DecimalFormat ("0.#");
                format.setMaximumFractionDigits(_numDigits - dDigits);
                format.setDecimalSeparatorAlwaysShown(false);
            } // dDigits
        } // d
        return result.append(format.format(d));
    } // format

    /** returns a string that is larger than any number that will be produced.
     *  Useful for planning formatting
     *  @return a string that will take up no less space than any number produced
     *  by {@link #format(double)}
     *  Assumes that "#" is larger than any digit.
     */
    public String maxString(){
        // assumes exponents are less than 3 digits
        StringBuffer result = new StringBuffer("-#.E-###");
        for (int i = 0; i < _numDigits; i++) result.append("#");
        return result.toString();
    }
    
    /** static method that uses the default number of digits
     *  @param d the number to format
     *  @see #format(double) for the details of formatting
     *  @return a String representation of the number
     */
    public static String formatNumber (double d){
        Num2Str format = new Num2Str();
        return format.format(d).toString();
    }
    
    /** static method that uses the default number of digits
     *  @param value an Object (that must be a Number) to format
     *  @see #format(double) for the details of formatting
     *  @return a String representation of the number
     *  @throws ClassCastException if value is not a Number
     */
    public static String formatNumber (Object value){
        double d = ((Number)value).doubleValue();
        Num2Str format = new Num2Str();
        return format.format(d).toString();
    } // formatNumber
    
} // Num2Str