/* StringFiler.java
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */

package util;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import javax.swing.JFileChooser;

/** Encapsulates saving and loading strings from a file. Saving is done in its own Thread. Loading a string
 *  is in the current Thread, since returning a value is more complex across Threads
 *  (but see {@link #loadFromThread()}).
 *  StringFiler itself should be used from the event thread.
 */
public class StringFiler{
    private String _text;
    private File _file;

    /** create a new instance of StringFiler with no filename
     *  @param text the text to save */
    public StringFiler (String text) {this(text, null);}

    /** create a new instance of StringFiler
     *  @param text the text to save (use null to use this to load a string, or (better) use the
     *  static method {link #loadString(String)})
     *  @param name the name of the file
     */
    public StringFiler (String text, String name){
        _text = text;
        if (name != null) _file = new File(name);
    } // constructor

    /** @return the name of the file (full path name).
     *  returns null if the file is not set */
    public String getFilename() {
        if (_file != null) return _file.getPath();
        return null;
    } // getFileName
    
    /** @return the name of the file (without the whole path).
     *  returns null if the file is not set
     */
    public String getShortName(){
        if (_file != null) return _file.getName();
        return null;
    } // getShortName

    /** @param name the name of the file */
    public void setFilename(String name) {
        if (name != null){
            _file = new File(name);
        }else{
            _file = null;
        }
    }

    /** @return the current text */
    public String getText() {return _text;}

    /** @param text the new text */
    public void setText (String text) {_text = text;}

    /** save the current text in the current file. If the file is not set, put up a save as dialog */
    public void save() {
        if (_file == null){
            saveAs();
        }else{
            save (_text, _file);
        } // if
     } // save
                

    /** changes the current text and saves it in the current file.
     * If the file is not set, put up a save as dialog 
     * @param text the new text to save
     */
    public void save (String text){
        _text = text;
        save();
    } // save

    /** save the current text in a file to be selected, and changes the file name with a save as dialog.
     *  if the user cancels from the dialog, does nothing
     */
    public void saveAs(){
        File file = getSaveFile (_file);
        if (file != null){
            _file = file;
            save();
        } // if
    } // saveAs

    /** changes the current text and saves it in a file to be selected, and changes the file name with a save as dialog.
     *  if the user cancels from the dialog, just changes the text but doesn't save anything
     */
    public void saveAs (String text){
        _text = text;
        saveAs();
    } // saveAs

    /** static method to just save without creating an object
     *  @param text the text to save
     *  @param file the File to save into (use null to put up a save as dialog)
     *  saves in its own thread. If the user cancels from the save as dialog, 
     *  does nothing
     */
    public static void save (String text, File file){
        if (file == null) file = getSaveFile(null);
        if (file == null) return;
        try{
            final byte[] b = text.getBytes();
            final File finalFile = file;
            Runnable r = new Runnable(){
                public void run(){
                    FileOutputStream fos = null;
                    try{
                        fos = new FileOutputStream(finalFile);
                        fos.write (b);
                    }catch (Exception ex1){
                        Object[] o = {finalFile.getPath()};
                        ErrorDialog.errorDialogSpecific ("SaveError", o, ex1);
                    }finally{ try{
                        fos.close();
                    }catch (Exception ex2){
                        // do nothing
                    }} // try
                } // run
            }; // new Runnable
            new Thread(r).start();
        }catch (Exception ex){
            Object[] o = {file.getPath()};
            ErrorDialog.errorDialogSpecific ("SaveError", o, ex);
            return;
        } // try
    } // save

    protected static File getSaveFile (File file){
        JFileChooser chooser = new JFileChooser(file);
        int returnVal = chooser.showSaveDialog(null);
        if(returnVal == JFileChooser.APPROVE_OPTION){
            return chooser.getSelectedFile();
        }
        return null;
    } // getSaveFile

    /** loads a string from a file. If {@link #getFilename()} == null, puts up an open file dialog
     *  if the user does not cancel, {@link #getFilename()} will return the name
     *  if the load is successful, {@link #getText()} will return the string loaded
     *  @return the string loaded; if the user cancelled or an error happened, return null
     */
    public String load() {
        if (_file == null){
            File file = getLoadFile(null);
            if (file == null) return null;
            _file = file;
        } // if
        String text = load (_file);
        if (text != null) _text = text;
        return text;
     } // load

    /** loads a string from a file in its own thread. If {@link #getFilename()} == null,
     * puts up an open file dialog
     *  if the user does not cancel, {@link #getFilename()} will return the name
     *  if the load is successful, {@link #getText()} will return the string loaded.
     *  if the load is not successful, {@link #getText()} will return null.
     *  @return the Thread that was started, for you to join() or whatever. If the user
     *  cancelled, returns null.
     */
     public Thread loadFromThread(){
         if (_file == null){
            File file = getLoadFile (null);
            if (file == null) return null;
            _file = file;
        } // if
        final File file = _file;
        Runnable r = new Runnable(){
            public void run(){
                String text = load (file);
                _text = text; // an atomic operation, one hopes
            } // run
        }; // new Runnable
        Thread t = new Thread(r);
        t.start();
        return t;
    } // loadFromThread

    /** static method to load a string from a file without creating an object.
     *  @param file the File to load from (use null to put up an open file dialog). If the user
     *  cancels or an error happens, returns null. An empty file returns an empty string
     */
    public static String load (File file){
        if (file == null) file = getLoadFile (null);
        if (file == null) return null;
        String text = null;
        try{
            FileInputStream fis = new FileInputStream(file);
            int size = fis.available();
            byte[] b = new byte[size];
            fis.read(b);
            text = new String (b);
        }catch (Exception ex){
            Object[] o = {file.getPath()};
            ErrorDialog.errorDialogSpecific ("LoadError", o, ex);
        } // try
        return text;
    } // load

    protected static File getLoadFile(File file){
        java.awt.Frame f = (java.awt.Frame) java.awt.KeyboardFocusManager.
          getCurrentKeyboardFocusManager().getActiveWindow();
        // probably should be localized; leave that for later
        JFileChooser chooser = new JFileChooser(file);
        int returnVal = chooser.showOpenDialog(f);
        if(returnVal == JFileChooser.APPROVE_OPTION) {
            return chooser.getSelectedFile();
        }
        return null;
    } // load
} // StringFiler