/*
 * StringPrinter.java
 *
 * Created on March 30, 2004, 7:25 AM
 *
 *  Copyright 2004 Daniel Wachsstock
 *  The contents of this file are subject to the Sun Public License
 *  Version 1.0 (the License); you may not use this file except in
 *  compliance with the License. A copy of the License is available at
 *  http://www.sun.com/ or http://www.geocities.com/tenua4java/license.html
 */
package util;

import java.awt.*;
import java.awt.print.*;
import javax.swing.JOptionPane;
import java.util.*;

/** Implements a simple printing class for a text string in a Thread.
 *  Based on a JavaWorld article by Michael Shoffner.
 *  Rather inefficient, since it repaginates far more often than it needs to,
 *  but would be appropriate for the kind of toy compiler I am writing.
 *
 * @author  Daniel Wachsstock
 */
public class StringPrinter  extends Thread implements Pageable, Printable {
   
    private String _text;
    private String _title;
    private Font _font;
    private PageFormat _format;
    private ArrayList _pages;

    /** Creates a new instance of StringPrinter with default Font and PageFormat */
    public StringPrinter (String text) {this(text, null, null);}
    /** Creates a new instance of StringPrinter with default PageFormat */
    public StringPrinter (String text, Font font) {this(text, font, null);}
    /** Creates a new instance of StringPrinter with default Font */
    public StringPrinter (String text, PageFormat format) {this(text, null, format);}
    /** Creates a new instance of StringPrinter.
     *  @param text the text to print
     *  @param font the font; uses monospaced 12-point if null
     *  @param format the PageFormat to use; uses PrinterJob.getPrinterJob().defaultPage()
     *  if null
     */
    public StringPrinter (String text, Font font, PageFormat format){
        _text = text;
        _font = font;
        if (_font == null) _font = new Font("Monospaced", Font.PLAIN, 12);
        _format = format;
    } // constructor

    /** prints the string */
    public void run(){
        PrinterJob job = PrinterJob.getPrinterJob();
        Book book = new Book();
        if (_format == null) _format = job.defaultPage();
        job.setPageable (this);
        if (job.printDialog()){
            try{
                job.print();
            }catch (PrinterException e){
                ErrorDialog.errorDialog("PrintError", e);
            } // try
        } // if
    } // run

    /** sets the title string 
     *  @param title the String to use for the header, along with the page number 
     */
    public void setTitle (String title){
        _title = title;
    } //  setTitle

    
    /** returns the title string
     *  @return the title string
     */
    public String getTitle (){
        return _title;
    } // title

    /** calculates the total number of printable pages
     *  @return the number of pages of the whole document
     */
    public int getNumberOfPages (){
        if (_format == null) return 1; // with no information, assume it all fits
        ArrayList pages = repaginate();
        return pages.size();
    } // countPages

    /** returns the PageFormat for a given page
     *  @param pageIndex the page to format
     *  @return a PageFormat for that page. All of them are the same
     *  @throws IndexOutOfBoundsException if pageIndex is not a valid page
     */
    public PageFormat getPageFormat(int pageIndex) throws IndexOutOfBoundsException {
        if (pageIndex < 0 || pageIndex > getNumberOfPages()) throw new IndexOutOfBoundsException();
        return _format;
    } // getPageFormat

    /** returns the Printable for a given page
     *  @param pageIndex the page that will be printed
     *  @return this, as a Printable interface
     *  @throws IndexOutOfBoundsException if pageIndex is not a valid page
     */
    public Printable getPrintable(int pageIndex) throws IndexOutOfBoundsException {
        if (pageIndex < 0 || pageIndex > getNumberOfPages()) throw new IndexOutOfBoundsException();
        return this;
    } // getPrintable
        
    /** renders a page
     *  @param g the Graphics to render on
     *  @param format the PageFormat to use for rendering
     *  @param pageIndex the page to print
     *  @return NO_SUCH_PAGE or PAGE_EXISTS, as approprite for pageIndex
     *  @throws IndexOutOfBoundsException if pageIndex is not a valid page
     */
    public int print(Graphics g, PageFormat format, int pageIndex) throws PrinterException{
    if (_format != format){
            _format = format;
            _pages = repaginate();
    } // if
    if (_pages == null) _pages = repaginate();
    if (pageIndex >= _pages.size()){
        return Printable.NO_SUCH_PAGE;
    } // if
    g.setFont (_font);
    g.setColor (Color.black);
    renderPage (g, pageIndex);
    return Printable.PAGE_EXISTS;
    } // print

    /** changes tabs to spaces. Inefficient; uses String.split(String)
     *  to turn the entire string into an array of Strings, then
     *  split's each one of those.
     *  @param s the string to detabify. Splits lines on '\n' only,
     *  not '\r'.
     *  @param tabSize the number of spaces to use for each tab
     *  @param truncate true if each tab field should be truncated
     *  to fit in tabSize-1 columns (the -1 ensures a space between
     *  tabs; false to keep the text intact and move to the next
     *  available tab stop. Strings get truncated from the end; numbers
     *  are rounded with {@link Num2Str} if possible, and replaced
     *  with '*'s (a la FORTRAN) if not.
     */
    public static StringBuffer untabify (String s, int tabSize, boolean truncate){
        String[] lines = s.split("\n");
        StringBuffer result = new StringBuffer();
        for (int line = 0; line < lines.length; line++){
            String[] tabs = lines[line].split ("\t");
            for (int tab = 0; tab < tabs.length; tab++){
                result.append (untabifyOneTab (tabs[tab], tabSize, truncate));
            } // for tab
            result.append ('\n');
        } // for line
        return result;
    } // untabify

    // adds spaces or truncates a string to fit in a tab stop
    static protected StringBuffer untabifyOneTab
      (String s, int tabSize, boolean truncate){
        int len = s.length();
        StringBuffer result = new StringBuffer(s);
        int targetSize = truncate ? tabSize :
          len + tabSize - len%tabSize;
        if (len > targetSize){
            truncate (result, targetSize);
        }else{
            // add spaces
            for (int i = 0; i < targetSize-len; i++) result.append(" ");
        } // if
        return result;
    } // untabifyOneTab

    // trucate a StringBuffer, with intelligent truncation for numbers
    static protected void truncate (StringBuffer result, int size){
        try{
            double d = Double.parseDouble (result.toString());
            // it's a number! Guess at how many digits to round to
            int digits = size-1;
            while (result.length() >= size && digits > 0){
                // keep trying to round as little as possible
                Num2Str formatter = new Num2Str (digits);
                result.delete (0, result.length());
                result.append(formatter.format(d));
                digits--; // try a smaller number
            } // while
            if (digits <= 0){
                // could not format it to fit
                result.delete (0, result.length());
                 while (result.length() < size-1 ) result.append("*");  
            }else{
                // make sure it didn't get too short
                 while (result.length() < size-1 ) result.append(" ");  
            } // if
        }catch (Exception ex){
            // it's not a number (this is a misuse of Exception catching
            // since this is a common case, but I don't know any other way
            // test for valid numbers (unless I do my own expression
            // matching))
            result.delete (size-1, result.length());
         }finally{
            result.append (" "); // at least one space at the end
         }
    } // truncate

    // creates an array of arrays of lines of text, one array per page
    protected ArrayList repaginate(){
        int pageHeight;
        if (_format != null){
                pageHeight = (int) _format.getImageableHeight();
        }else{
                pageHeight = Integer.MAX_VALUE;
        } // if

        int lineHeight = _font.getSize();

        ArrayList pages = new ArrayList();
        ArrayList page = new ArrayList();

        int currentHeight = addHeader (page, pages.size()+1);

        String[] splitText = _text.split("\\n");
        for (int i = 0; i < splitText.length; i++){
            if (currentHeight + lineHeight > pageHeight){
                // accumulate the current page
                pages.add(page);
                // and start a new one
                page = new ArrayList();
                currentHeight = addHeader (page, pages.size()+1);
            } // if
            page.add (splitText[i]);
            currentHeight += lineHeight;
        } // for
        pages.add (page);
        return pages;
    } // repaginate

    // adds a header line to page and returns the number of pixels that the header took up
    protected int addHeader (ArrayList page, int num){
            String line = "Page " + num;
            if (_title != null) line = _title + "   " + line;
            page.add(line);
            page.add(""); // blank line
            return _font.getSize()*2;
    } // addHeader

    // draws one page onto g
    protected void renderPage (Graphics g, int index){            
            int x = 0;
            int y0 = 0;
            if (_format != null){
                    x = (int) _format.getImageableX();
                    y0 = (int) _format.getImageableY();
            } // if
            int y = y0 + _font.getSize();
            Iterator i = ((ArrayList) _pages.get(index)).iterator();
            while (i.hasNext()){
                String line = (String) i.next();
                g.drawString (line, x, y);
                y += _font.getSize();
            } // while
    } // renderPage

} // StringPrinter
